/*requires Schema.Administration.sql*/

DECLARE @ProcedureSchema NVARCHAR(256);
DECLARE @ProcedureName   NVARCHAR(256);

SET @ProcedureSchema = 'Administration' ;
SET @ProcedureName = 'CheckRecoveryModelToEnvAdequation' ;

RAISERROR('-----------------------------------------------------------------------------------------------------------------',0,1);
RAISERROR('PROCEDURE [%s].[%s]',0,1,@ProcedureSchema,@ProcedureName);

IF  NOT EXISTS (SELECT 1 FROM sys.objects WHERE object_id = OBJECT_ID(N'[' + @ProcedureSchema + N'].[' + @ProcedureName +  N']') AND type in (N'P'))
BEGIN
    BEGIN TRY
        EXECUTE ('CREATE Procedure [' + @ProcedureSchema + '].[' + @ProcedureName +  '] ( ' +
                ' @ServerName    varchar(512), ' +
                ' @DbName    varchar(50) ' +
                ') ' +
                'AS ' +
                'BEGIN ' +
                '   SELECT ''Not implemented'' ' +
                'END')
    END TRY
    BEGIN CATCH
        PRINT '   Error while trying to create procedure'
        RETURN
    END CATCH

    PRINT '   PROCEDURE created.'
END
GO

ALTER PROCEDURE [Administration].[CheckRecoveryModelToEnvAdequation] (
    @ExecutionMode                  VARCHAR(64)  = 'REPORT', -- Possible values: REPORT, MONITORING, CORRECT
    @WithPageVerifyCheck            BIT          = 1,
    @DefaultRecoveryMode4Production VARCHAR(256) = 'FULL',
    @Debug                          SMALLINT     = 0
)
AS
/*
  ===================================================================================
    DESCRIPTION:
        Gets the environment of current server from a record in Common.ApplicationParams table.
        This record is defined using the actual SQL Server Instance Name (with hostname) as ApplicationName
        and 'Environment' as ParamName.
        
        These values must be defined at server install using Finish-SQLServerSetup PoSH script (TODO)

    PARAMETERS:
    
        @Debug      Tells the stored procedure whether to be more talkative or not.
                    0 means no debug output
                    1 means some debug output (but no T-SQL statement)
                    2 or more means everything

    REQUIREMENTS:

    EXAMPLE USAGE :
    
        EXEC [Administration].[CheckRecoveryModelToEnvAdequation] @Debug = 2
        
        EXEC [Administration].[CheckRecoveryModelToEnvAdequation] @Debug = 2, @ExecutionMode = 'MONITORING'
        
        EXEC [Administration].[CheckRecoveryModelToEnvAdequation] @Debug = 2, @ExecutionMode = 'CORRECT'
  ===================================================================================
*/
BEGIN
    SET NOCOUNT ON;
    DECLARE @ExecRet            INT;
    DECLARE @tsql               nvarchar(max);
    DECLARE @LineFeed           CHAR(2);
    DECLARE @ProcedureName      VARCHAR(1024);
    DECLARE @FullInstanceName	VARCHAR(1024);
    DECLARE @SoftwareEnv		VARCHAR(256);
    DECLARE @FilterDbList       VARCHAR(MAX);
	DECLARE @LogMsg             VARCHAR(MAX);
    
    -- for @ExecutionMode = "CORRECT"
    DECLARE @CurrentDbName  VARCHAR(256);
    DECLARE @CurOutcome     VARCHAR(64);
    DECLARE @ErrorNumber    INT;
    DECLARE @ErrorLine      INT;
    DECLARE @ErrorMessage   NVARCHAR(4000);
    DECLARE @ErrorSeverity  INT;
    DECLARE @ErrorState     INT;
    

    SELECT
        @ProcedureName      = QUOTENAME(OBJECT_SCHEMA_NAME(@@PROCID)) + '.' + QUOTENAME(OBJECT_NAME(@@PROCID)),
        @tsql               = '',
        @LineFeed           = CHAR(13) + CHAR(10),
        @Debug              = CASE WHEN @Debug > 2 THEN 2 ELSE @Debug END,
        @ExecutionMode      = UPPER(LTRIM(RTRIM(@ExecutionMode)))
    ;

    if (@Debug >= 1)
    BEGIN
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
        RAISERROR('-- Now running %s stored procedure.',0,1,@ProcedureName);
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
    END;
    
    IF(@ExecutionMode NOT IN ('REPORT','MONITORING','CORRECT'))
    BEGIN
        RAISERROR('Invalid value for parameter @ExecutionMode: [%s]',12,1,@ExecutionMode);
        RETURN;
    END;
    
    IF(@Debug > 0)
    BEGIN
        RAISERROR('-- Getting back environment related informations',0,1);
    END;
    
    select 
		@FullInstanceName = CAST(SERVERPROPERTY('ServerName') AS VARCHAR(256)) + 
							CASE 
								WHEN SERVERPROPERTY('InstanceName') IS NULL THEN '' 
								ELSE 
									CHAR(92) /* backslash*/ + CAST(SERVERPROPERTY('InstanceName') AS VARCHAR(256)) 
							END
    ;

    EXEC Common.getApplicationparam @ApplicationName = @FullInstanceName, @ParamName = 'SystemDatabases', @ParamValue = @FilterDbList OUTPUT;
    EXEC Common.getApplicationparam @ApplicationName = @FullInstanceName, @ParamName = 'Environment',     @ParamValue = @SoftwareEnv  OUTPUT;
    
    SET @FilterDbList = CASE WHEN LEN(LTRIM(RTRIM(@FilterDbList))) = 0 THEN NULL ELSE @FilterDbList END;
    
    IF(@FilterDbList IS NOT NULL)
    BEGIN 
        SET @FilterDbList = '''' + REPLACE(@FilterDbList,',',''',''') + '''' ;
    END;
    
    IF(@Debug > 0)
    BEGIN 
        RAISERROR('-- Filter Db List value: %s',0,1,@FilterDbList);
    END;    
    
    IF(@Debug > 0)
    BEGIN
        RAISERROR('-- Creating summary temporary table',0,1);
    END;
    
    IF(OBJECT_ID('tempdb..#dbs') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #dbs';
    END;
    
    CREATE TABLE #dbs (
        DatabaseName        VARCHAR(256)    NOT NULL,
        UserAccess          VARCHAR(64)     NOT NULL,
        RecoveryModel       VARCHAR(64)     NOT NULL,
        PageVerifyOption    VARCHAR(64)     NOT NULL,
        CorrectiveStatement NVARCHAR(2048)  NULL,
        CorrectionOutcome   VARCHAR(32)     NULL,
        CorrectionLog       VARCHAR(4000)   NULL 
    );
    
    IF(@Debug > 0)
    BEGIN
        RAISERROR('-- Building T-SQL Query',0,1);
    END;
    
    SET @tsql = 'INSERT INTO #dbs (' + @LineFeed +
                '    DatabaseName,UserAccess,RecoveryModel,PageVerifyOption, CorrectiveStatement' + @LineFeed +
                ')' + @LineFeed +
                'SELECT' + @LineFeed +
                '    [name],' + @LineFeed +
                '    user_access_desc,' + @LineFeed +
                '    recovery_model_desc,' + @LineFeed +
                '    page_verify_option_desc,' + @LineFeed +
                '    ''USE [master];'' + CHAR(13) + CHAR(10) + ' + @LineFeed + 
                '    ''ALTER DATABASE '' + QUOTENAME([name]) + '' SET RECOVERY ' +  CASE 
                                                                                        WHEN @SoftwareEnv = 'Production' THEN @DefaultRecoveryMode4Production 
                                                                                        ELSE 'SIMPLE' 
                                                                                    END  +  ', PAGE_VERIFY CHECKSUM''' + @LineFeed +
                'FROM sys.databases' + @LineFeed +
                'WHERE state = 0 /*online*/' + @LineFeed +
                CASE 
                    WHEN @FilterDbList IS NULL THEN '' 
                    ELSE 'AND [name] NOT IN (' + @FilterDbList + ')' + @LineFeed
                END +
                'AND (' + @LineFeed + 
                '    recovery_model_desc ' +    
                    CASE    
                        WHEN @SoftwareEnv = 'Production' THEN 'NOT IN (''FULL'',''BULK_LOGGED'')'
                        ELSE ' <> ''SIMPLE'''
                    END + @LineFeed +
                CASE 
                    WHEN @WithPageVerifyCheck = 0 THEN ''
                    ELSE '    OR page_verify_option_desc <> ''CHECKSUM''' + @LineFeed
                END + 
                ');'
                ;
                
    IF(@Debug > 0)
    BEGIN
        RAISERROR('-- Running T-SQL query...',0,1);
    END;
    
    IF(@Debug = 2)
    BEGIN
        RAISERROR('------------------------- ( T-SQL Query ) ----------------------------',0,1);
        RAISERROR(@tsql,0,1);
        RAISERROR('----------------------------------------------------------------------',0,1);
    END;
    
    EXEC @ExecRet = sp_executesql @tsql;
    
    IF(@ExecRet <> 0)
    BEGIN
        RAISERROR('An error occurred during T-SQL Query execution',12,1);
        RETURN;
    END;
    
    IF(@ExecutionMode = 'REPORT')
    BEGIN
        SELECT * FROM #dbs;
    END;
    ELSE IF(@ExecutionMode = 'MONITORING')
    BEGIN
        IF((SELECT COUNT(*) FROM #Dbs) > 0)
        BEGIN 
            RAISERROR('One or more databases do not comply to standard',12,1);
            RETURN;
        END;
    END;
    ELSE    -- correct 
    BEGIN
        WHILE (1 = 1)
        BEGIN
            SET @CurrentDbName  = NULL;
            SET @tsql           = NULL;
            SET @CurOutcome     = NULL;
            
            SELECT TOP 1 
                @CurrentDbName = DatabaseName,
                @tsql          = CorrectiveStatement
            FROM #dbs 
            WHERE CorrectionOutcome IS NULL
            ;
            
            IF(@CurrentDbName IS NULL)
            BEGIN
                BREAK;
            END;
            
            IF(@Debug > 0)
            BEGIN
                RAISERROR('Now taking correcting action for database [%s]',0,1,@CurrentDbName);
            END;
            
            IF(@Debug = 2)
            BEGIN
                RAISERROR('------------------------- ( T-SQL Query ) ----------------------------',0,1);
                RAISERROR(@tsql,0,1);
                RAISERROR('----------------------------------------------------------------------',0,1);
            END;            
            
            BEGIN TRY 
                EXEC @ExecRet = sp_executesql @tsql ;
                
                IF(@ExecRet <> 0)
                BEGIN
                    RAISERROR('An error occurred during execution of corrective T-SQL statement',12,1);
                END;
                
                SET @CurOutcome = 'SUCCESS';
                
            END TRY
            BEGIN CATCH 

                SELECT 
                    @ErrorNumber    = ERROR_NUMBER(),
                    @ErrorLine      = ERROR_LINE(),
                    @ErrorMessage   = ERROR_MESSAGE(),
                    @ErrorSeverity  = ERROR_SEVERITY(),
                    @ErrorState     = ERROR_STATE()
                ;
             
                SET @LogMsg = 'Caught error #' + CAST(@ErrorNumber AS VARCHAR(10)) + /*'during XXX' + */ @LineFeed +
                              'At line #' + CAST(@ErrorLine AS VARCHAR(10)) + @LineFeed +
                              'With Severity ' + CAST(@ErrorSeverity AS VARCHAR(10)) + ' State ' + CAST(@ErrorState AS VARCHAR(10)) + @LineFeed +
                              @LineFeed +
                              'Message:' + @LineFeed +
                              '-------' + @LineFeed +
                              @ErrorMessage 
                              ;
                              
                IF(@Debug > 0)
                BEGIN
                    RAISERROR(@LogMsg,10,1);
                END;
                
                SET @CurOutcome = 'ERROR';
            END CATCH ;
            
            UPDATE #dbs 
            SET 
                CorrectionOutcome = @CurOutcome,
                CorrectionLog     = @LogMsg 
            WHERE DatabaseName = @CurrentDbName
            ;
        END;    
        
        SELECT * FROM #dbs;
        
        IF((SELECT COUNT(*) FROM #dbs WHERE CorrectionOutcome <> 'SUCCESS') > 0)
        BEGIN
            RAISERROR('One or more corrections did not succeed',12,1);
            RETURN;
        END;
    END;
    
    IF(@Debug > 0)
    BEGIN
        RAISERROR('-- Performing cleanups',0,1);
    END;
    
    IF(OBJECT_ID('tempdb..#dbs') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #dbs';
    END;
    
    
    if (@Debug >= 1)
    BEGIN
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
        RAISERROR('-- Execution of %s completed.',0,1,@ProcedureName);
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
    END;


END
GO


IF (@@ERROR = 0)
BEGIN
    PRINT '   PROCEDURE altered.';
END
ELSE
BEGIN
    PRINT '   Error while trying to alter procedure';
    RETURN
END;
GO

RAISERROR('-----------------------------------------------------------------------------------------------------------------',0,1);
RAISERROR('',0,1);
GO

