<#

.DESCRIPTION
This script will detect your Availability Group primary replica and copy all of its instance level objects to all of the secondary replicas within the Availability Group.

.EXAMPLE
./Sync-SqlInstanceObjects.ps1 -DomainName "mydomain.com" -Port 1433

.NOTES
One limitation of this script is that it assumes you only have one availability group.

.LINK
https://www.sqlhammer.com/synchronizing-server-objects-for-availability-groups/

#>

[CmdletBinding(SupportsShouldProcess=$true)]
Param
(
    [Parameter(Mandatory=$false,
			   ValueFromPipeline=$true,
			   ValueFromPipelineByPropertyName=$true,
			   HelpMessage="Domain that this Availability Group is in. I.e. 'mydomain.com'")]
	[string]$DomainName,
    [Parameter(Mandatory=$false,
			   ValueFromPipeline=$true,
			   ValueFromPipelineByPropertyName=$true,
			   HelpMessage="Port number that all connections will be made under.")]
	[bigint]$Port
)

Write-Output "Sync started."

#Error handling

$ErrorActionPreference = "stop";

Trap 
{

    $err = $_.Exception
    while ( $err.InnerException )
    {

        $err = $err.InnerException
        Write-Output $err.Message

    };

}

# Helper functions

function Get-FullConnectionName ($ServerName)
{

    return "$ServerName.$DomainName,$Port"

}

function Compare-SystemVersion ($Version1, $Version2)
{
    
    try
    {
        if ([System.Version]$Version1 -lt [System.Version]$Version2)
        {
            return -1
        }
        elseif ([System.Version]$Version1 -eq [System.Version]$Version2)
        {
            return 0
        }
        elseif ([System.Version]$Version1 -gt [System.Version]$Version2)
        {
            return 1
        }
    }
    catch
    {
        return -2
    }

}

# Prerequisites

try
{
    
    Write-Output "Valiating prerequisites."

	<# Works but needs to run as admin which our SQL Agent is not going to do.
	 # Code remains for reference of how to initiate a new node

    if ((Get-Module -ListAvailable -Name dbatools) -eq $null `
        -or ((Compare-SystemVersion (Get-Module -ListAvailable -Name dbatools).Version "0.9.24") -lt 0))
    {

        # Install the installer
        if ((Get-Module -ListAvailable -Name PowerShellGet) -eq $null)
        { 
            Install-PackageProvider -Name NuGet -MinimumVersion 2.8.5.201 -Force | Out-Null
        }

        Import-Module PowerShellGet | Out-Null

        # Install dbatools
        Install-Module dbatools -MinimumVersion 0.9.24 -AllowClobber -Force

    }
	#>

    if ((Get-Module -Name dbatools) -eq $null)
    {
    
	    Import-Module dbatools | Out-Null

    }

    Write-Output "Prerequisites loaded."

}
catch
{
    
    Write-Error $_.Exception.Message -EA Continue
    Write-Error "One or more of the prerequisites did not load. Review previous errors for more details." -EA Stop
    return

}

# Detect Availability Group Primary replica

Write-Output "Begin query of Availability Group state."

$DataSource = (Get-FullConnectionName $env:COMPUTERNAME)

$Query = @"
SELECT replica_server_name
	, HAGS.primary_replica
FROM sys.availability_replicas AR
INNER JOIN sys.dm_hadr_availability_group_states HAGS
INNER JOIN sys.availability_groups AG ON AG.group_id = HAGS.group_id
    ON HAGS.group_id = AR.group_id;
"@

$AGStates = Invoke-Sqlcmd2 -ServerInstance $DataSource -Query $Query -ConnectionTimeout 30

if(([DBNull]::Value).Equals($AGStates[0].primary_replica))
{

    Write-Error "Availability Group state query returned no results. Confirm that you connected to a SQL Server instance running an Availability Group. No work was accomplished."
    return

}

Write-Output "Completed query of Availability Group state."

# Only runs if it is the primary, secondaries remain idle.
if($AGStates[0].primary_replica.ToLower().CompareTo($env:COMPUTERNAME.ToLower()) -eq 0)
{

    foreach($replica in $AGStates)
    {
        # Skip this iteration because source and destination would be the same
        if($replica.replica_server_name.ToLower().CompareTo($env:COMPUTERNAME.ToLower()) -eq 0)
        {
            continue
        }

        $SecondaryReplica = (Get-FullConnectionName $replica.replica_server_name)

        Write-Output "Copying objects from $DataSource to $SecondaryReplica"

        # Copy objects

        Write-Output "Copying Logins."
        Copy-DbaLogin -Source $DataSource -Destination $SecondaryReplica

        Write-Output "Copying BackupDevices."
        Copy-DbaBackupDevice -Source $DataSource -Destination $SecondaryReplica

        Write-Output "Copying Endpoints."
        Copy-DbaEndpoint -Source $DataSource -Destination $SecondaryReplica

        Write-Output "Copying Linked Servers."
        Copy-DbaLinkedServer -Source $DataSource -Destination $SecondaryReplica

        Write-Output "Copying Agent Categories."
        Copy-DbaAgentCategory -Source $DataSource -Destination $SecondaryReplica

        Write-Output "Copying Shared Schedules."
        Copy-DbaAgentSharedSchedule -Source $DataSource -Destination $SecondaryReplica

        Write-Output "Copying Operators."
        Copy-DbaAgentOperator -Source $DataSource -Destination $SecondaryReplica

        Write-Output "Copying Credentials."
        Copy-DbaCredential -Source $DataSource -Destination $SecondaryReplica
        
        Write-Output "Copying Proxy Account."
        Copy-DbaAgentProxyAccount -Source $DataSource -Destination $SecondaryReplica
        
        Write-Output "Copying Job."
        Copy-DbaAgentJob -Source $DataSource -Destination $SecondaryReplica
        
        Write-Output "Copy complete from $DataSource to $SecondaryReplica"
    }

}
else
{
    Write-Output "No work accompished, this is not the primary node."
}

Write-Output "Sync complete."
