/*requires Schema.Administration.sql*/

DECLARE @ProcedureSchema NVARCHAR(256);
DECLARE @ProcedureName   NVARCHAR(256);

SET @ProcedureSchema = 'Administration' ;
SET @ProcedureName = 'DropDatabasePrincipal' ;

RAISERROR('-----------------------------------------------------------------------------------------------------------------',0,1);
RAISERROR('PROCEDURE [%s].[%s]',0,1,@ProcedureSchema,@ProcedureName);

IF  NOT EXISTS (SELECT 1 FROM sys.objects WHERE object_id = OBJECT_ID(N'[' + @ProcedureSchema + N'].[' + @ProcedureName +  N']') AND type in (N'P'))
BEGIN
    BEGIN TRY
        EXECUTE ('CREATE Procedure [' + @ProcedureSchema + '].[' + @ProcedureName +  '] ( ' +
                ' @ServerName    varchar(512), ' +
                ' @DatabaseName    varchar(50) ' +
                ') ' +
                'AS ' +
                'BEGIN ' +
                '   SELECT ''Not implemented'' ' +
                'END')
    END TRY
    BEGIN CATCH
        PRINT '   Error while trying to create procedure'
        RETURN
    END CATCH

    PRINT '   PROCEDURE created.'
END
GO



	
ALTER PROCEDURE [Administration].[DropDatabasePrincipal] (
    @DatabaseName                       VARCHAR(256),
    @PrincipalName                      VARCHAR(256),
    @PreserveAssignedPermissions        BIT             = 1,
    @AssignRolePermissionsToItsMembers  BIT             = 0,
    @AlterDbObjectsOwnership            BIT             = 1, 
    @NewDbObjectOwner                   VARCHAR(256)    = 'dbo',
    @WithLog                            BIT             = 1, -- TODO: use it
    @RunCheckOnly                       BIT             = 1,
    @Debug                              BIT             = 0
)
AS
/*
  ===================================================================================
    DESCRIPTION:

    PARAMETERS:
    
        -- NewDbObjectOwner => reset to schema/objects ownership to @NewDbObjectOwner 

    REQUIREMENTS:

    EXAMPLE USAGE :
        
        EXEC [Administration].[DropDatabasePrincipal]
				@DatabaseName				= 'TestDb',
				@PrincipalName				= 'RoleToBeDropped',
				--@NewPermissionAssigner	= 'dbo',
				@AlterDbObjectsOwnership	= 1,
				@NewDbObjectOwner			= 'dbo',
				@Debug						= 1
		;

    
        EXEC [Administration].[DropDatabasePrincipal]
				@DatabaseName						= 'TestDb',
				@PrincipalName						= 'RoleToBeDropped',
				@AssignRolePermissionsToItsMembers	= 1,
				--@NewPermissionAssigner				= 'dbo',
				@AlterDbObjectsOwnership			= 1,
				@NewDbObjectOwner					= 'dbo',
				@RunCheckOnly                       = 0,
				@Debug								= 1
		;
  ===================================================================================
*/
BEGIN
    SET NOCOUNT ON;
    DECLARE @tsql                       nvarchar(max);
    DECLARE @LineFeed                   CHAR(2);
    DECLARE @ProcedureName              VARCHAR(1024);
	DECLARE @LogMsg                     VARCHAR(8000);
	DECLARE @ExecRet                    INT;
	DECLARE @TmpCnt                     BIGINT;
	DECLARE @TmpINT                     BIGINT;
    
        
    DECLARE @QuotedDatabaseName         VARCHAR(260);
    DECLARE @QuotedDbPrincipal          VARCHAR(260);
    DECLARE @PrincipalIsUser            BIT;
    DECLARE @CurRecordName              VARCHAR(256);
    DECLARE @CurObjectType              VARCHAR(256);
    DECLARE @CurRecordId                BIGINT;

    SELECT
        @ProcedureName              = QUOTENAME(OBJECT_SCHEMA_NAME(@@PROCID)) + '.' + QUOTENAME(OBJECT_NAME(@@PROCID)),
        @DatabaseName               = ISNULL(LTRIM(RTRIM(@DatabaseName)),DB_NAME()),
        @PrincipalName              = CASE WHEN LEN(LTRIM(RTRIM(@PrincipalName))) = 0           THEN NULL ELSE @PrincipalName END,
        @NewDbObjectOwner           = CASE WHEN LEN(LTRIM(RTRIM(@NewDbObjectOwner))) = 0        THEN NULL ELSE @NewDbObjectOwner END,
        @QuotedDatabaseName         = QUOTENAME(@DatabaseName),
        @QuotedDbPrincipal          = QUOTENAME(@PrincipalName),
        @tsql                       = '',
        @LineFeed                   = CHAR(13) + CHAR(10)
    ;

    if (@Debug = 1)
    BEGIN
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
        RAISERROR('-- Now running %s stored procedure.',0,1,@ProcedureName);
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
    END;
    
    if(@Debug = 1)
    BEGIN
        RAISERROR('-- Performing some prerequisities checks and parameter validation',0,1);
    END;

    IF(DB_ID(@DatabaseName) IS NULL)
    BEGIN
        RAISERROR('Database with name [%s] cannot be found on server',12,1,@DatabaseName);
        RETURN;
    END;
    
    IF(@PrincipalName IS NULL)
    BEGIN
        RAISERROR('Provide a database principal name',12,1);
        RETURN;
    END;
    
    SET @tsql = 'USE ' + @QuotedDatabaseName + ';' + @LineFeed +
                'select ' + @LineFeed +
                '    @retBit = CASE ' + @LineFeed +
                '                  WHEN [type] = ''R'' THEN 0 ' + @LineFeed +
                '                  ELSE 1' + @LineFeed +
                '              END' + @LineFeed +
                'FROM sys.database_principals' + @LineFeed +
                'WHERE name = @PName' + @LineFeed +
                ';'
                ;            
    
    IF(@Debug = 1)
    BEGIN
		SET @LogMsg = '/* Next Query to run:' + @LineFeed + @tsql + @LineFeed + '*/';
        RAISERROR(@LogMsg,0,1);
    END;
    
    
    SET @PrincipalIsUser = NULL;
    exec @ExecRet = sp_executesql 
                            @tsql , 
                            N'@retBit BIT OUTPUT,@PName VARCHAR(256)' , 
                            @retBit = @PrincipalIsUser OUTPUT, 
                            @PName  = @PrincipalName
    ;
    -- check execution success
    IF(@ExecRet <> 0 OR @@ERROR <> 0)
    BEGIN
        RAISERROR('An internal error occurred when getting back informations about database principal',12,1);
        RETURN;
    END;

    IF(@PrincipalIsUser IS NULL)
    BEGIN
        RAISERROR('Principal with name [%s] not found in database [%s]',12,1,@PrincipalName,@DatabaseName);
        RETURN;
    END;
    ELSE
    BEGIN
        IF(@Debug = 1)
        BEGIN 
            SET @LogMsg = 'Principal name found as a database ' + CASE WHEN @PrincipalIsUser = 0 THEN 'ROLE' ELSE 'USER' END ;
            RAISERROR(@LogMsg,0,1);
        END;
    END;
    
    if(@Debug = 1)
    BEGIN
        RAISERROR('-- Now, we are sure the database principal exists',0,1);
        RAISERROR('-- Checking principal [%s] is allowed for removal',0,1,@PrincipalName);
    END;   
    
    WITH MSShippedDbPrincipals (
        PrincipalName,
        isUser
    )
    AS (
        SELECT 'dbo',1
        UNION ALL
        SELECT 'guest',1
        UNION ALL
        SELECT 'INFORMATION_SCHEMA',1
        UNION ALL
        SELECT 'sys',1
        UNION ALL
        SELECT 'public',0
        UNION ALL
        SELECT 'db_securityadmin',0
        UNION ALL
        SELECT 'db_owner',0
        UNION ALL
        SELECT 'db_denydatawriter',0
        UNION ALL
        SELECT 'db_denydatareader',0
        UNION ALL
        SELECT 'db_ddladmin',0
        UNION ALL
        SELECT 'db_datawriter',0
        UNION ALL
        SELECT 'db_datareader',0
        UNION ALL
        SELECT 'db_backupoperator',0
        UNION ALL
        SELECT 'db_accessadmin',0
        UNION ALL
        SELECT distinct service_account,1
        FROM   master.sys.dm_server_services
    )
    SELECT @TmpCnt = COUNT(*)
    FROM MSShippedDbPrincipals 
    WHERE PrincipalName = @PrincipalName 
    AND isUser = @PrincipalIsUser
    ;
    
    if(@TmpCnt > 0) 
    BEGIN
        RAISERROR('Database principal cannot be dropped.',14,1) WITH NOWAIT;
        RETURN;
    END;
    
    IF(@AlterDbObjectsOwnership = 1)
    BEGIN
        IF(@Debug = 1)
        BEGIN
            RAISERROR('Checking @NewDbObjectOwner as we need to use the value of this parameter',0,1);
        END;
        
        IF(@NewDbObjectOwner IS NULL)
        BEGIN
            RAISERROR('A value for parameter @NewDbObjectOwner is mandatory.',12,1);
            RETURN;
        END;
        
        IF(@NewDbObjectOwner = @PrincipalName)
        BEGIN
            RAISERROR('Provide a value for parameter @NewDbObjectOwner that is different from the one you used for @PrincipalName parameter.',12,1);
            RETURN;
        END;
        
        SET @tsql = 'USE ' + @QuotedDatabaseName + ';' + @LineFeed +
                    'SELECT @prpId = USER_ID(@PName);'
                    ;
                    
        IF(@Debug = 1)
        BEGIN
            SET @LogMsg = '/* Next Query to run:' + @LineFeed + @tsql + @LineFeed + '*/';
            RAISERROR(@LogMsg,0,1);
        END;
                    
        EXEC @ExecRet = sp_executesql 
                            @tsql , 
                            N'@prpId INT OUTPUT,@PName VARCHAR(256)',
                            @prpId  = @TmpInt OUTPUT,
                            @PName  = @NewDbObjectOwner 
        ;
        
        IF(@ExecRet <> 0 OR @@ERROR <> 0)
        BEGIN
            RAISERROR('An internal error occurred when getting back informations about database principal',12,1);
            RETURN;
        END;

        IF(@TmpInt IS NULL)
        BEGIN
            RAISERROR('Principal with name [%s] not found in database [%s]',12,1,@NewDbObjectOwner,@DatabaseName);
            RETURN;
        END;
    END;    
    
    
    IF(@PrincipalIsUser = 1)
    BEGIN
        /*
            Avoid following error message :
            
            Msg 15136, Level 16, State 1, Line 2
            The database principal is set as the execution context of one or more procedures, functions, or event notifications and cannot be dropped.
        */
        
        if(@Debug = 1)
        BEGIN 
            RAISERROR('Checking that the user is not set as the execution context of one or more programmability components in database.',0,1);
        END;
        
        SET @tsql = 'USE ' + @QuotedDatabaseName + ';' + @LineFeed +
                    'SELECT @cnt = COUNT(*) FROM sys.sql_modules where execute_as_principal_id = user_id(@UserName)';
            
        exec @ExecRet = sp_executesql @tsql, N'@cnt INT OUTPUT,@UserName SYSNAME',@cnt = @TmpCnt OUTPUT,@UserName = @PrincipalName ;
        
		IF(@ExecRet <> 0 OR @@ERROR <> 0)
		BEGIN
			RAISERROR('An internal error occurred while checking for contextual references to [%s] database principal',12,1,@PrincipalName);
			RETURN;
		END;

        IF(@TmpCnt > 0) 
        BEGIN 
            SET @LogMsg = 'In database ' + @QuotedDatabaseName + ', the database principal ' + @QuotedDbPrincipal + ' is set as the execution context of one or more procedures, functions, or event notifications and cannot be dropped.' + @LineFeed +
                            'List of modules:' + @LineFeed;
            
            -- Getting back the list
            SET @tsql = 'USE ' + @QuotedDatabaseName + ';' + @LineFeed +
                        'SELECT @moduleList += ''    '' + OBJECT_SCHEMA_NAME(object_id) + ''.'' + OBJECT_NAME(object_id) + CHAR(13) + CHAR(10) FROM sys.sql_modules where execute_as_principal_id = user_id(@UserName)';
            
            exec @ExecRet = sp_executesql @tsql, N'@moduleList NVARCHAR(MAX) OUTPUT,@UserName SYSNAME',@moduleList = @LogMsg OUTPUT,@UserName = @PrincipalName ;    
            RAISERROR(@LogMsg,14,1) WITH NOWAIT;
            RETURN;
        END;
    END;
    
    if(@Debug = 1)
    BEGIN
        RAISERROR('-- Parameters validation success.',0,1);
    END;  

    if(@Debug = 1)
    BEGIN
        RAISERROR('-- Getting back permissions assigned by database principal',0,1);
    END;
    
    IF(OBJECT_ID('tempdb..#AssignedPermissions') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #AssignedPermissions';
    END;
    
    CREATE TABLE #AssignedPermissions (
        Id4AP       INT IDENTITY(1,1),
        ReAssignSql NVARCHAR(MAX)
    );
    
    SET @tsql = 'USE ' + @QuotedDatabaseName + ';' + @LineFeed +
                'INSERT INTO #AssignedPermissions (' + @LineFeed +
                '    ReAssignSql' + @LineFeed +
                ')' + @LineFeed +
                'select ' + @LineFeed + 
                'CASE ' + @LineFeed +
                '    WHEN state_desc = ''GRANT_WITH_GRANT_OPTION'' THEN ''GRANT'' ' + @LineFeed +
                '    ELSE state_desc ' + @LineFeed +
                'END + '' '' + permission_name + '' ON '' + ' + @LineFeed +
                'CASE ' + @LineFeed +
                '    WHEN class_desc = ''OBJECT_OR_COLUMN'' THEN' + @LineFeed +
                '        ''OBJECT::'' + QUOTENAME(OBJECT_SCHEMA_NAME(major_id)) + ''.'' + QUOTENAME(OBJECT_NAME(major_id))' + @LineFeed +
                '    WHEN class_desc = ''DATABASE'' THEN' + @LineFeed +
                '        ''DATABASE::'' + DB_NAME() ' + @LineFeed +
                '    WHEN class_desc = ''SCHEMA'' THEN' + @LineFeed +
                '        ''SCHEMA::'' + QUOTENAME(OBJECT_SCHEMA_NAME(major_id))' + @LineFeed +
                '    ELSE ''/*TODO: handle other cases - at the moment, this statement will fail*/''' + @LineFeed +
                'END +' + @LineFeed +
                ''' TO '' + QUOTENAME(USER_NAME(grantee_principal_id)) + ' + @LineFeed +
                'CASE ' + @LineFeed +
                '    WHEN state_desc = ''GRANT_WITH_GRANT_OPTION'' THEN '' WITH GRANT OPTION''' + @LineFeed +
                '    ELSE ''''' + @LineFeed +
                'END + '';''' + @LineFeed +
                'FROM sys.database_permissions ' + @LineFeed +
                'WHERE grantor_principal_id = USER_ID(@PrincipalName)' + @LineFeed +
                ';'
                ;
    
    IF(@Debug = 1)
    BEGIN
        SET @LogMsg = '/* Next Query to run:' + @LineFeed + @tsql + @LineFeed + '*/';
        RAISERROR(@LogMsg,0,1);
    END;
                
    EXEC @ExecRet = sp_executesql 
                        @tsql , 
                        N'@PrincipalName VARCHAR(256)',
                        @PrincipalName          = @PrincipalName
    ;
    
    IF(@ExecRet <> 0 OR @@ERROR <> 0)
    BEGIN
        RAISERROR('An internal error occurred when getting back informations about database principal',12,1);
        RETURN;
    END;
    
    IF(@PreserveAssignedPermissions = 0)
    BEGIN
        IF(@Debug = 1)
        BEGIN
            RAISERROR('-- Checking no permissions are assigned by [%s] database principal',0,1,@PrincipalName);
        END;
        
        SELECT @TmpCnt = COUNT_BIG(*) 
        FROM #AssignedPermissions
        ;
        
        IF(@TmpCnt > 0)
        BEGIN
            RAISERROR('Permissions are assigned by database principal [%s]. Consider setting @PreserveAssignedPermissions parameter is set to 1.',12,1,@PrincipalName);
            RETURN;
        END;
    END;
    
    
    if(@Debug = 1)
    BEGIN
        RAISERROR('-- Getting back the list of objects owned by database principal [%s]',0,1,@PrincipalName);
    END;
    
    IF(OBJECT_ID('tempdb..#OwnedDbObject') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #OwnedDbObject';
    END;
    
    CREATE TABLE #OwnedDbObject (
        ODbObjId   INT IDENTITY(1,1),
        ObjectName VARCHAR(256),
        ObjectType VARCHAR(16)
    );
    
    if(@Debug = 1)
    BEGIN
        RAISERROR('--    > Schemas',0,1);
    END;
    
    SET @tsql = 'USE [' + @DatabaseName + ']; ' + @LineFeed +
                'INSERT INTO #OwnedDbObject (' + @LineFeed +
                '    ObjectName,ObjectType' + @LineFeed +
                ')' + @LineFeed +
                'select' + @LineFeed +
                '    name, ' + @LineFeed +
                '    ''SCHEMA'' ' + @LineFeed +
                'from sys.schemas ' + @LineFeed +
                'where principal_id = USER_ID(@PrincipalName);' + @LineFeed 
                ;
    
    IF(@Debug = 1)
    BEGIN
		SET @LogMsg = '/* Next Query to run:' + @LineFeed + @tsql + @LineFeed + '*/';
        RAISERROR(@LogMsg,0,1);
    END;
    
    exec sp_executesql @tsql, N'@PrincipalName VARCHAR(256)', @PrincipalName = @PrincipalName;
    
    SELECT @TmpCnt = COUNT_BIG(*) FROM #OwnedDbObject where ObjectType = 'SCHEMA';
    
    if((@TmpCnt > 0) AND @AlterDbObjectsOwnership = 0)
    BEGIN 
		SET @LogMsg = 'Database principal ' + @QuotedDbPrincipal + ' owns ' + CONVERT(VARCHAR(10),@TmpCnt) + ' schemas(s) in the database ' + @QuotedDatabaseName + '.' ;
        RAISERROR (@LogMsg,12,1) WITH NOWAIT;
        RETURN;
    END;  
    
    if(@Debug = 1)
    BEGIN
        RAISERROR('--    > Database Roles',0,1);
    END;    
    
    SET @tsql = 'USE [' + @DatabaseName + ']; ' + @LineFeed +
                'INSERT INTO #OwnedDbObject' + @LineFeed +
                'select' + @LineFeed +
                '    name, ' + @LineFeed +
                '    ''ROLE'' ' + @LineFeed +
                'from sys.database_principals ' + @LineFeed +
                'where type=''R'' and owning_principal_id = USER_ID(@PrincipalName);' + @LineFeed 
                ;
    
    IF(@Debug = 1)
    BEGIN
		SET @LogMsg = '/* Next Query to run:' + @LineFeed + @tsql + @LineFeed + '*/';
        RAISERROR(@LogMsg,0,1);
    END;
    
    exec sp_executesql @tsql, N'@PrincipalName VARCHAR(256)', @PrincipalName = @PrincipalName;  
    
    SELECT @TmpCnt = COUNT_BIG(*) FROM #OwnedDbObject where ObjectType = 'ROLE';
    
    if((@TmpCnt > 0) AND @AlterDbObjectsOwnership = 0)
    BEGIN 
		SET @LogMsg = 'Database principal ' + @QuotedDbPrincipal + ' owns ' + CONVERT(VARCHAR(10),@TmpCnt) + ' roles(s) in the database ' + @QuotedDatabaseName + '.' ;
        RAISERROR (@LogMsg,12,1) WITH NOWAIT;
        RETURN;
    END;    
    
    if(@Debug = 1)
    BEGIN
        RAISERROR('-- Collecting roles memberships for database principal [%s]',0,1,@PrincipalName);
    END;
        
    SET @tsql = 'USE [' + @DatabaseName + ']; ' + @LineFeed +
                'INSERT INTO #OwnedDbObject' + @LineFeed +
                'select' + @LineFeed +
                '    dp.name, ' + @LineFeed +
                '    ''MEMBERSHIP'' ' + @LineFeed +
                'from sys.database_role_members drm'  + @LineFeed +
                'inner join sys.database_principals dp '  + @LineFeed +
                'on dp.principal_id= drm.role_principal_id'  + @LineFeed +
                'where member_principal_id = USER_ID(@PrincipalName);' + @LineFeed 
                ;
    
    IF(@Debug = 1)
    BEGIN
		SET @LogMsg = '/* Next Query to run:' + @LineFeed + @tsql + @LineFeed + '*/';
        RAISERROR(@LogMsg,0,1);
    END;
    
    exec sp_executesql @tsql, N'@PrincipalName VARCHAR(256)', @PrincipalName = @PrincipalName;
    
    IF(OBJECT_ID('tempdb..#RoleMembers') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #RoleMembers';
    END;
    
    CREATE TABLE #RoleMembers (
        MemberName VARCHAR(256) NOT NULL
    );
    
    IF(OBJECT_ID('tempdb..#RolePermissions') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #RolePermissions';
    END;
    
    CREATE TABLE #RolePermissions (
        RPid        INT IDENTITY(1,1),    
        AssignSql   NVARCHAR(MAX)
    );
    
    IF(@PrincipalIsUser = 0)
    BEGIN
        if(@Debug = 1)
        BEGIN
            RAISERROR('-- Collecting roles members for database role [%s]',0,1,@PrincipalName);
        END;        
        
        SET @tsql = 'USE ' + @QuotedDatabaseName + ';' + @LineFeed +
                    'INSERT INTO #RoleMembers' + @LineFeed +
                    'select' + @LineFeed +
                    '    USER_NAME(member_principal_id) ' + @LineFeed +
                    'FROM sys.database_role_members ' + @LineFeed +
                    'where role_principal_id = USER_ID(@PrincipalName)' + @LineFeed +
                    ';'
                    ;
        
        IF(@Debug = 1)
        BEGIN
            SET @LogMsg = '/* Next Query to run:' + @LineFeed + @tsql + @LineFeed + '*/';
            RAISERROR(@LogMsg,0,1);
        END;
        
        exec sp_executesql @tsql, N'@PrincipalName VARCHAR(256)', @PrincipalName = @PrincipalName;  
        
        IF(@AssignRolePermissionsToItsMembers  = 1)
        BEGIN
            if(@Debug = 1)
            BEGIN
                RAISERROR('-- Collecting permissions directly assigned to database role [%s]',0,1,@PrincipalName);
            END;        
            
            SET @tsql = 'USE ' + @QuotedDatabaseName + ';' + @LineFeed +
                        'INSERT INTO #RolePermissions (' + @LineFeed +
                        '    AssignSql' + @LineFeed +
                        ')' + @LineFeed +
                        'select' + @LineFeed +
                        'CASE ' + @LineFeed +
                        '    WHEN state_desc = ''GRANT_WITH_GRANT_OPTION'' THEN ''GRANT'' ' + @LineFeed +
                        '    ELSE state_desc ' + @LineFeed +
                        'END + '' '' + permission_name + '' ON '' + ' + @LineFeed +
                        'CASE ' + @LineFeed +
                        '    WHEN class_desc = ''OBJECT_OR_COLUMN'' THEN' + @LineFeed +
                        '        ''OBJECT::'' + QUOTENAME(OBJECT_SCHEMA_NAME(major_id)) + ''.'' + QUOTENAME(OBJECT_NAME(major_id))' + @LineFeed +
                        '    WHEN class_desc = ''DATABASE'' THEN' + @LineFeed +
                        '        ''DATABASE::'' + DB_NAME() ' + @LineFeed +
                        '    WHEN class_desc = ''SCHEMA'' THEN' + @LineFeed +
                        '        ''SCHEMA::'' + QUOTENAME(OBJECT_SCHEMA_NAME(major_id))' + @LineFeed +
                        '    ELSE ''/*TODO: handle other cases - at the moment, this statement will fail*/''' + @LineFeed +
                        'END +' + @LineFeed +
                        ''' TO <RoleMember> '' +' + @LineFeed +
                        'CASE ' + @LineFeed +
                        '    WHEN state_desc = ''GRANT_WITH_GRANT_OPTION'' THEN '' WITH GRANT OPTION''' + @LineFeed +
                        '    ELSE ''''' + @LineFeed +
                        'END +' + @LineFeed +
                        ''';''' + @LineFeed +
                        'FROM sys.database_permissions ' + @LineFeed +
                        'where grantee_principal_id = USER_ID(@PrincipalName)' + @LineFeed +
                        ';'
                        ;
            
            IF(@Debug = 1)
            BEGIN
                SET @LogMsg = '/* Next Query to run:' + @LineFeed + @tsql + @LineFeed + '*/';
                RAISERROR(@LogMsg,0,1);
            END;
            
            exec @ExecRet = sp_executesql 
						@tsql, 
						N'@PrincipalName VARCHAR(256)', 
						@PrincipalName		   = @PrincipalName
			; 
            
            IF(@ExecRet <> 0 OR @@ERROR <> 0)
            BEGIN
                RAISERROR('An internal error occurred when getting back permissions directly assigned to database role [%s]',12,1,@PrincipalName);
                RETURN;
            END;
        END;
    END;
    
    IF(@Debug = 1)
    BEGIN
        RAISERROR('Every checks and collections have been performed successfully.',0,1);
    END;
    
    IF(@RunCheckOnly = 1)
    BEGIN
        IF(@Debug = 1)
        BEGIN
            RAISERROR('This run was done with @RunCheckOnly parameter set to 1.',0,1);
        END;
        GOTO cleanups;
    END;
    
    -- =*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*
    IF(@Debug = 1)
    BEGIN
        RAISERROR('We can actually perform the action...',0,1);
    END;
    -- =*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*
    
    
    
    IF(@AlterDbObjectsOwnership = 1)
    BEGIN
        if(@Debug = 1)
        BEGIN
            RAISERROR('-- Changing object owner from [%s] to [%s]',0,1,@PrincipalName,@NewDbObjectOwner);
        END;
        
        WHILE (1 = 1)
        BEGIN 
            SET @CurRecordName = NULL;
            
            SELECT TOP 1
                @TmpCnt         = ODbObjId,
                @CurRecordName  = ObjectName,
                @CurObjectType  = ObjectType
            FROM #OwnedDbObject
            ;
            
            IF(@CurRecordName IS NULL)
            BEGIN
                BREAK;
            END;
            
            IF(@Debug = 1)
            BEGIN
                RAISERROR('    > Type: %s | Name: %s',0,1,@CurObjectType,@CurRecordName);
            END;
            
            if(@CurObjectType = 'MEMBERSHIP')
            BEGIN 
                SET @tsql = 'USE [' + @DatabaseName + ']; ' + @LineFeed + 
                            'EXEC sp_droprolemember N''' + @CurRecordName + ''', N''' + @PrincipalName + ''';'  + @LineFeed
                            ;
            END;
            ELSE 
            BEGIN 
                SET @tsql = 'USE [' + @DatabaseName + ']; ' + @LineFeed + 
                            'ALTER AUTHORIZATION ON ' + @CurObjectType + '::'+QUOTENAME(@CurRecordName) + ' TO [' + @NewDbObjectOwner + '];'  
                            ;
            END;
            
            IF(@Debug = 1)
            BEGIN
                SET @LogMsg = '/* Next Query to run:' + @LineFeed + @tsql + @LineFeed + '*/';
                RAISERROR(@LogMsg,0,1);
            END;
            
            exec @ExecRet = sp_executesql @tsql;
            
            IF(@ExecRet <> 0 OR @@ERROR <> 0)
            BEGIN
                RAISERROR('Unable to assign permission',12,1);
                RETURN;
            END;
            
            DELETE FROM #OwnedDbObject WHERE ODbObjId = @TmpCnt;
        END;
    END;
    
    IF(@PrincipalIsUser = 0)
    BEGIN
        
        IF(@Debug = 1)
        BEGIN
            RAISERROR('-- Removing members from [%s] database role',0,1,@PrincipalName);
        END;
        
        IF(@AssignRolePermissionsToItsMembers = 1 AND @Debug = 1)
        BEGIN 
            RAISERROR('-- At the same time, assigning [%s]''s permissions to these members',0,1,@PrincipalName);
        END;
        
        WHILE(1=1)
        BEGIN
            SET @CurRecordName = NULL;
            SELECT TOP 1
                @CurRecordName = MemberName
            FROM #RoleMembers
            ;
            
            -- exit condition
            IF(@CurRecordName IS NULL)
            BEGIN
                BREAK;
            END;
            
            IF(@Debug = 1)
            BEGIN
                RAISERROR('--     > [%s]',0,1,@CurRecordName);
            END;
            
            SET @tsql = 'USE ' + @QuotedDatabaseName + ';' + @LineFeed +
                        'ALTER ROLE '+ @QuotedDbPrincipal +' DROP MEMBER '+ QUOTENAME(@CurRecordName) + ';';
            
            IF(@Debug = 1)
            BEGIN
                SET @LogMsg = '/* Next Query to run:' + @LineFeed + @tsql + @LineFeed + '*/';
                RAISERROR(@LogMsg,0,1);
            END;
            
            exec sp_executesql @tsql;
            
            IF(@AssignRolePermissionsToItsMembers = 1)
            BEGIN
                SELECT @CurRecordId = MIN(RPid)
                FROM #RolePermissions
                ;
                
                WHILE(@CurRecordId IS NOT NULL)
                BEGIN
                    SELECT @tsql = REPLACE(AssignSql,'<RoleMember>',QUOTENAME(@CurRecordName))
                    FROM #RolePermissions
                    WHERE RPid = @CurRecordId
                    ;
                    
                    SET @tsql = 'USE ' + @QuotedDatabaseName + ';' + @LineFeed +
                                @tsql
                                ;
                    
                    IF(@Debug = 1)
                    BEGIN
                        SET @LogMsg = '/* Next Query to run:' + @LineFeed + @tsql + @LineFeed + '*/';
                        RAISERROR(@LogMsg,0,1);
                    END;
                    
                    exec sp_executesql @tsql ;
                        
                    -- prepare for next iteration    
                    SELECT @CurRecordId = MIN(RPid)
                    FROM #RolePermissions
                    WHERE RPid > @CurRecordId
                    ;
                END;    
                
            END;
            
            -- preparing for next iteration
            DELETE 
            FROM #RoleMembers
            WHERE MemberName = @CurRecordName
            ;
        END;
    END;
    
    
    SET @tsql = 'USE ' + @QuotedDatabaseName + ';' + @LineFeed +
                'DROP ROLE ' + @QuotedDbPrincipal + ';' 
                ;
    IF(@Debug = 1)
    BEGIN
        SET @LogMsg = '/* Next Query to run:' + @LineFeed + @tsql + @LineFeed + '*/';
        RAISERROR(@LogMsg,0,1);
    END;      
    
    EXEC @ExecRet = sp_executesql @tsql ;
    
    IF(@ExecRet <> 0 OR @@ERROR <> 0)
    BEGIN
        RAISERROR('An error occurred during role drop',12,1);
        RETURN;
    END;
    
    IF(@PreserveAssignedPermissions = 1)
    BEGIN
    
        if(@Debug = 1)
        BEGIN
            RAISERROR('-- Setting back permissions assigned by [%s]',0,1,@PrincipalName);
        END;
        
        WHILE (1 = 1)
        BEGIN
            SET @TmpCnt = NULL;
            
            SELECT @TmpCnt = MIN(Id4AP)
            FROM #AssignedPermissions
            ;
            
            IF(@TmpCnt IS NULL)
            BEGIN
                BREAK;
            END;    
            
            -- assign 
            SELECT @tsql = ReAssignSql
            FROM #AssignedPermissions
            WHERE Id4AP = @TmpCnt 
            ;
            
            SET @tsql = 'USE ' + @QuotedDatabaseName + ';' + @LineFeed +
                        @tsql
                        ;
                        
            IF(@Debug = 1)
            BEGIN
                SET @LogMsg = '/* Next Query to run:' + @LineFeed + @tsql + @LineFeed + '*/';
                RAISERROR(@LogMsg,0,1);
            END;
            
            exec @ExecRet = sp_executesql @tsql, N'@PrincipalName VARCHAR(256)', @PrincipalName = @PrincipalName;
            
            IF(@ExecRet <> 0 OR @@ERROR <> 0)
            BEGIN
                RAISERROR('Unable to assign permission',12,1);
                RETURN;
            END;
            
            -- carry on 
            DELETE FROM #AssignedPermissions
            WHERE Id4AP = @TmpCnt
            ;
        END;
    END;
    
    cleanups:
    IF(@Debug = 1)
    BEGIN
        RAISERROR('Performing cleanups...',0,1);
    END;
    
    IF(OBJECT_ID('tempdb..#AssignedPermissions') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #AssignedPermissions';
    END;
    
    IF(OBJECT_ID('tempdb..#OwnedDbObject') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #OwnedDbObject';
    END;
    
    IF(OBJECT_ID('tempdb..#RoleMembers') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #RoleMembers';
    END;
    
    IF(OBJECT_ID('tempdb..#RolePermissions') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #RolePermissions';
    END;    
    
    if (@Debug = 1)
    BEGIN
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
        RAISERROR('-- Execution of %s completed.',0,1,@ProcedureName);
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
    END;


END
GO


IF (@@ERROR = 0)
BEGIN
    PRINT '   PROCEDURE altered.';
END
ELSE
BEGIN
    PRINT '   Error while trying to alter procedure';
    RETURN
END;
GO

RAISERROR('-----------------------------------------------------------------------------------------------------------------',0,1);
RAISERROR('',0,1);
GO

