/*requires Schema.Reporting.sql*/
/*requires Procedure.Monitoring.CollectDeadlockInformation.sql*/
/*requires Procedure.Reporting.ShredDeadlockHistoryTbl.sql*/

/*

    This script creates a SQL Server Agent Job when no job with its name exists.
    
    The job created collect Deadlock XML graphs from an Extended Events Session to a table.
    Then it will split each XML graph into a set of records and store the results into another table.
            
    To Edit settings, look for following string: "-- <# USER SETTINGS"
*/

-- Temporary tables creation
-- ============================================

IF(OBJECT_ID('tempdb..#StepDefaultValuesMapping') IS NOT NULL)
BEGIN
    EXEC sp_executesql N'DROP TABLE #StepDefaultValuesMapping;';
END;

CREATE TABLE #StepDefaultValuesMapping (
    CreationParamName   VARCHAR(128),
    TableValue          VARCHAR(128),
    ActualValue         INT
);

IF(OBJECT_ID('tempdb..#JobCreationCommands') IS NOT NULL)
BEGIN
    EXEC sp_executesql N'DROP TABLE #JobCreationCommands;';
END;

CREATE TABLE #JobCreationCommands (
    OrderId             INT IDENTITY(1,1),
    Comments            VARCHAR(4000),
    Stmt                NVARCHAR(MAX)
);

IF(OBJECT_ID('tempdb..#JobSchedules') IS NOT NULL)
BEGIN
    EXEC sp_executesql N'DROP TABLE #JobSchedules;';
END;

CREATE TABLE #JobSchedules (
    ScheduleId                  INT IDENTITY(1,1),
    ScheduleName                VARCHAR(256)        NOT NULL,
    ScheduleFrequency           VARCHAR(32)         NOT NULL DEFAULT 'DEFAULT', -- DEFAULT, ONCE, DAILY, WEEKLY, MONTHLY, MONTHLY_INTRVAL, AT_AGENT_START, ON_IDLE_CONDITION
    ScheduleFreqIntrvalAsDay    VARCHAR(32)         NULL, -- SUNDAY,MONDAY,TUESDAY,WEDNESDAY,THURSDAY, FRIDAY,SATURDAY
    ScheduleFreqIntrvalAsNumber INT                 NULL,
    ScheduleFreqIntrvalAsRange  VARCHAR(32)         NULL, -- SUNDAY,MONDAY,TUESDAY,WEDNESDAY,THURSDAY, FRIDAY,SATURDAY,DAY, WEEKDAY,WEEKEND
    ScheduleFreqSubDayType      VARCHAR(32)         NULL, -- AT_THIS_TIME,MINUTE,HOUR
    ScheduleFreqSubDayIntrval   INT                 NOT NULL DEFAULT 0,
    ScheduleFreqRelativeIntrval VARCHAR(32)         NULL, -- FIRST, SECOND,THIRD,FOURTH,LAST
    ScheduleFreqRecurrence      INT                 NOT NULL DEFAULT 0,
    ScheduleActivationDateTime  DATETIME2           NOT NULL DEFAULT SYSDATETIME(),
    ScheduleEndDateTime         DATETIME2           NULL
);

INSERT INTO #StepDefaultValuesMapping
VALUES 
    ('OnSuccessAction','NEXT',3),
    ('OnSuccessAction','QUIT_FAILURE',2),
    ('OnSuccessAction','QUIT_SUCCESS',1),
    ('OnSuccessAction','GOTO',4),
    ('OnFailureAction','NEXT',3),
    ('OnFailureAction','QUIT_FAILURE',2),
    ('OnFailureAction','QUIT_SUCCESS',1),
    ('OnFailureAction','GOTO',4),
    ('BehaviorOptions','OVERWRITE_FILE',0),
    ('BehaviorOptions','APPEND_FILE',2),
    ('BehaviorOptions','TSQL_TO_STEP_HISTORY',4),
    ('BehaviorOptions','OVERWRITE_TABLE',8),
    ('BehaviorOptions','APPEND_TABLE',16),
    ('BehaviorOptions','ALL_TO_JOB_HISTORY',32),
    ('BehaviorOptions','WINDOWS_EVENT_SIGNAL',64),
    ('ScheduleFrequency','DEFAULT',0),
    ('ScheduleFrequency','ONCE',1),
    ('ScheduleFrequency','DAILY',4),
    ('ScheduleFrequency','WEEKLY',8),
    ('ScheduleFrequency','MONTHLY',16),
    ('ScheduleFrequency','MONTHLY_INTRVAL',32),
    ('ScheduleFrequency','AT_AGENT_START',64),
    ('ScheduleFrequency','ON_IDLE_CONDITION',128),
    ('ScheduleFreqIntrvalAsDay','SUNDAY',1),
    ('ScheduleFreqIntrvalAsDay','MONDAY',2),
    ('ScheduleFreqIntrvalAsDay','TUESDAY',4),
    ('ScheduleFreqIntrvalAsDay','WEDNESDAY',8),
    ('ScheduleFreqIntrvalAsDay','THURSDAY',16),
    ('ScheduleFreqIntrvalAsDay','FRIDAY',32),
    ('ScheduleFreqIntrvalAsDay','SATURDAY',64),
    ('ScheduleFreqIntrvalAsRange','SUNDAY',1),
    ('ScheduleFreqIntrvalAsRange','MONDAY',2),
    ('ScheduleFreqIntrvalAsRange','TUESDAY',3),
    ('ScheduleFreqIntrvalAsRange','WEDNESDAY',4),
    ('ScheduleFreqIntrvalAsRange','THURSDAY',5),
    ('ScheduleFreqIntrvalAsRange','FRIDAY',6),
    ('ScheduleFreqIntrvalAsRange','SATURDAY',7),
    ('ScheduleFreqIntrvalAsRange','DAY',8),
    ('ScheduleFreqIntrvalAsRange','WEEKDAY',9),
    ('ScheduleFreqIntrvalAsRange','WEEKEND',10),
    ('ScheduleFreqSubDayType','AT_THIS_TIME',1),
    ('ScheduleFreqSubDayType','MINUTE',4),
    ('ScheduleFreqSubDayType','HOUR',8),
    ('ScheduleFreqRelativeIntrval','FIRST',1),
    ('ScheduleFreqRelativeIntrval','SECOND',2),
    ('ScheduleFreqRelativeIntrval','THIRD',4),
    ('ScheduleFreqRelativeIntrval','FOURTH',8),
    ('ScheduleFreqRelativeIntrval','LAST',16)
;

IF(OBJECT_ID('tempdb..#AgentJobSteps') IS NOT NULL)
BEGIN
    EXEC sp_executesql N'DROP TABLE #AgentJobSteps;';
END;

CREATE TABLE #AgentJobSteps (
    StepId                  INT             NOT NULL    IDENTITY(1,1),
    StepName                VARCHAR(256)    NOT NULL,
    StepLanguage            VARCHAR(128)    NOT NULL,
    StepText                VARCHAR(MAX)    NOT NULL,
    ExpectedReturnedVal     INT             NOT NULL DEFAULT 0,
    OnSuccessAction         VARCHAR(128)    NOT NULL DEFAULT 'NEXT',            -- Expected Values : NEXT, QUIT_SUCCESS,QUIT_FAILURE,GOTO
    OnSuccessStepId         INT             NOT NULL DEFAULT 0,
    OnFailureAction         VARCHAR(128)    NOT NULL DEFAULT 'QUIT_FAILURE',    -- Expected Values : NEXT, QUIT_SUCCESS,QUIT_FAILURE,GOTO
    OnFailureStepId         INT             NULL,
    DatabaseName            VARCHAR(256)    NOT NULL DEFAULT 'master',
    DatabaseUserName        VARCHAR(256)    NULL,
    RetryAttemptsCount      INT             NOT NULL DEFAULT 0,
    RetryInterval           INT             NOT NULL DEFAULT 0, -- in minutes
    OutputFileName          VARCHAR(256)    NULL,
    BehaviorOptions         VARCHAR(256)    NOT NULL DEFAULT 'OVERWRITE_FILE',  -- Expected Values : OVERWRITE_FILE, APPEND_FILE, TSQL_TO_STEP_HISTORY,
                                                                                --                   OVERWRITE_TABLE, APPEND_TABLE,ALL_TO_JOB_HISTORY,WINDOWS_EVENT_SIGNAL
    ProxyId                 INT             NULL,
    ProxyName               VARCHAR(256)    NULL
);

-- Variables declaration
-- ============================================

-- General variables
DECLARE @tsql                   NVARCHAR(MAX);
DECLARE @tsql_vars              NVARCHAR(4000);
DECLARE @LogMsg                 VARCHAR(4000);
DECLARE @LineFeed               CHAR(2);
DECLARE @CurrentDbName          VARCHAR(256);
DECLARE @RunWithDebugOutput     BIT;
DECLARE @ReturnCode             INT;
DECLARE @InnerReturnCode        INT;
DECLARE @currentCommandId       INT;
DECLARE @currentStepComments    VARCHAR(4000);

-- Agent Job related variables
DECLARE @AgentJobName           NVARCHAR(1024);
DECLARE @AgentJobCategory       NVARCHAR(1024);
DECLARE @AgentJobDescription    NVARCHAR(4000);
DECLARE @AgentJobOwner          NVARCHAR(1024);
DECLARE @NotifyEmailOperator    NVARCHAR(1024);
DECLARE @NotifyByEmail          BIT;
DECLARE @NotifyByEmailIntVal    INT;
DECLARE @NotifyToEventLog       BIT;
DECLARE @NotifyToEventLogIntVal INT;
DECLARE @CreateJobEnabled       BIT;
DECLARE @AgentJobId             uniqueidentifier;


-- Agent Job step variables (for loop on #AgentJobSteps)
DECLARE @BehavioralFlag         INT;
DECLARE @EventSessionName       VARCHAR(256);

-- Agent Job schedule variables
DECLARE @JobScheduleId          INT;


-- Process related variables
DECLARE @TargetDatabaseName		VARCHAR(256);
DECLARE @HistorySchemaName		VARCHAR(256);
DECLARE @HistoryTableName		VARCHAR(256);
DECLARE @ShreddingSchemaName    VARCHAR(256);
DECLARE @ShreddingTableName     VARCHAR(256);


--
-- Initializations
-- ============================================

SELECT 
    @LineFeed               = CHAR(13) + CHAR(10),
    @ReturnCode             = 0,
    @NotifyByEmailIntVal    = 0,
    @NotifyToEventLogIntVal = 0,
    @CurrentDbName          = DB_NAME()
;



-- ------------------------------------------------------------------------------------------------------
-- <# USER SETTINGS 

SELECT 
    -- Related to Agent Job  
    @AgentJobName           = N'[Monitoring] Historize Deadlock Data' ,
    @AgentJobCategory       = N'Database Monitoring',
    @AgentJobDescription    = N'A job that collects and transforms data about deadlock events',
    @NotifyByEmail          = 1,
    @NotifyEmailOperator    = N'The DBA Team',
    @NotifyToEventLog       = 1,
    @CreateJobEnabled       = 0, 
    @AgentJobOwner          = 'sa',
    @RunWithDebugOutput     = 1,         -- Set this to 1 when you want this script to be more talkative
	@TargetDatabaseName		= DB_NAME(),
    
    -- Related to Process
	@HistorySchemaName      = 'Monitoring',
	@HistoryTableName       = 'DeadlocksHistory',
	@ShreddingSchemaName    = 'Reporting',
	@ShreddingTableName     = 'ShreddedDeadlocksHistory',
    @EventSessionName       = 'Collect-Deadlocks'
;

WITH Data2Insert (
    StepId,
    StepName,
    StepLanguage,
    StepText,
    ExpectedReturnedVal,
    OnSuccessAction,
    OnSuccessStepId,
    OnFailureAction,
    OnFailureStepId,
    DatabaseName,
    DatabaseUserName,
    RetryAttemptsCount,
    RetryInterval,
    OutputFileName,
    BehaviorOptions,
    ProxyId,
    ProxyName
)
AS (
    SELECT NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,'master',NULL,NULL,NULL,NULL,NULL,NULL,NULL
    UNION ALL 
    SELECT 
        1, 
        'Data Collection',
        'TSQL',
        'EXEC [Monitoring].[CollectDeadlockInformation]' + @LineFeed +
        '        @OutputType                 = ''NONE'',' + @LineFeed +
        '        @UseApplicationParamsTable  = 1,' + @LineFeed +
        '        @EventSessionName           = ''' + @EventSessionName + ''',' + @LineFeed +
        '        @ApplicationName            = ''Day2Day Database Management'',' + @LineFeed +
        '        @OutputDatabaseName         = ''' + @CurrentDbName + ''',' + @LineFeed +
        '        @OutputSchemaName           = ''' + @HistorySchemaName + ''',' + @LineFeed +
        '        @OutputTableName            = ''' + @HistoryTableName  + ''',' + @LineFeed +
        '        @Debug                      = 0' + @LineFeed +
        ';',
        0,
        'NEXT',
        0,
        'QUIT_FAILURE',
        0,
        @CurrentDbName,
        NULL,
        0,
        0,
        NULL,
        'TSQL_TO_STEP_HISTORY',
        NULL,
        NULL
    UNION ALL
    SELECT 
        1, 
        'Data Transformation',
        'TSQL',
        'EXEC [Reporting].[ShredDeadlockHistoryTbl]' + @LineFeed + 
        '        @SourceSchemaName  = ''' + @HistorySchemaName + ''',' + @LineFeed +
        '        @SourceTableName   = ''' + @HistoryTableName + ''',' + @LineFeed +
        '        @TargetSchemaName  = ''' + @ShreddingSchemaName + ''',' + @LineFeed +
        '        @TargetTableName   = ''' + @ShreddingTableName + ''',' + @LineFeed +
        '        @Debug		        = 0' + @LineFeed +
        ';',
        0,
        'NEXT',
        0,
        'QUIT_FAILURE',
        0,
        @CurrentDbName,
        NULL,
        0,
        0,
        NULL,
        'TSQL_TO_STEP_HISTORY',
        NULL,
        NULL
)
INSERT INTO #AgentJobSteps (
    StepName,
    StepLanguage,
    StepText,
    ExpectedReturnedVal,
    OnSuccessAction,
    OnSuccessStepId,
    OnFailureAction,
    OnFailureStepId,
    DatabaseName,
    DatabaseUserName,
    RetryAttemptsCount,
    RetryInterval,
    OutputFileName,
    BehaviorOptions,
    ProxyId,
    ProxyName
)
SELECT
    StepName,
    StepLanguage,
    StepText,
    ExpectedReturnedVal,
    OnSuccessAction,
    OnSuccessStepId,
    OnFailureAction,
    OnFailureStepId,
    DatabaseName,
    DatabaseUserName,
    RetryAttemptsCount,
    RetryInterval,
    OutputFileName,
    BehaviorOptions,
    ProxyId,
    ProxyName
FROM Data2Insert
WHERE StepId IS NOT NULL
ORDER BY StepId ASC
;

WITH Data2Insert (
    ScheduleName,
    ScheduleFrequency,
    ScheduleFreqIntrvalAsDay,
    ScheduleFreqIntrvalAsNumber,
    ScheduleFreqIntrvalAsRange,
    ScheduleFreqSubDayType,
    ScheduleFreqSubDayIntrval,
    ScheduleFreqRelativeIntrval,
    ScheduleFreqRecurrence,
    ScheduleActivationDateTime,
    ScheduleEndDateTime
)
AS (
    SELECT 
        NULL, NULL,NULL,NULL,NULL,NULL, NULL,NULL,NULL,NULL,NULL
    UNION ALL 
    SELECT 
        -- Runs every (1) day every 1 hour @XX:42:36
        'Deadlock-Monitoring-Schedule',
        'DAILY',
        NULL,
        1,
        NULL,
        'HOUR',
        1,
        NULL,
        0,
        CONVERT(DATETIME2,  CONVERT(CHAR(4),DATEPART(YEAR,SYSDATETIME())) + '-' + CONVERT(CHAR(2),DATEPART(MONTH,SYSDATETIME())) + '-' + CONVERT(CHAR(2),DATEPART(DAY,SYSDATETIME())) + ' ' + '00:42:36', 120),
		NULL
)
INSERT INTO #JobSchedules (
    ScheduleName,
    ScheduleFrequency,
    ScheduleFreqIntrvalAsDay,
    ScheduleFreqIntrvalAsNumber,
    ScheduleFreqIntrvalAsRange,
    ScheduleFreqSubDayType,
    ScheduleFreqSubDayIntrval,
    ScheduleFreqRelativeIntrval,
    ScheduleFreqRecurrence,
    ScheduleActivationDateTime,
    ScheduleEndDateTime
)
SELECT *
FROM Data2Insert 
WHERE ScheduleName IS NOT NULL 
;


-- #> USER SETTINGS 
-- ------------------------------------------------------------------------------------------------------



--
-- Actual Script start. DO NO EDIT BELOW
-- ============================================

-- ensure it's ok with user input for schedules

update #JobSchedules
set ScheduleFreqIntrvalAsNumber = 1
where ScheduleFrequency = 'DAY'
and (
     ScheduleFreqIntrvalAsNumber IS NULL 
  OR ScheduleFreqIntrvalAsNumber < 1
);



IF(@RunWithDebugOutput = 1)
BEGIN
    RAISERROR('Checking availability of SQL Server Agent',0,1);
END;


IF (NOT EXISTS (SELECT 1 FROM master.dbo.sysprocesses WHERE program_name = N'SQLAgent - Generic Refresher'))
BEGIN
    RAISERROR('Error: SQL Agent process not found',12,1) WITH NOWAIT;
	RAISERROR('       Unable to setup SQL Agent job ',0,1) WITH NOWAIT;
    GOTO QuitWithRollback;
END;


IF(@RunWithDebugOutput = 1)
BEGIN
    RAISERROR('SQL Agent process found up and running',0,1) WITH NOWAIT;
END;


IF(@RunWithDebugOutput = 1)
BEGIN
    RAISERROR('Checking user-defined parameters',0,1);
END;

IF(LEN(LTRIM(RTRIM(@NotifyEmailOperator))) = 0)
BEGIN
    SET @NotifyEmailOperator = NULL;
    SET @NotifyByEmail       = 0;
END;

IF(@NotifyByEmail = 1)
BEGIN
    SET @NotifyByEmailIntVal = 2; -- Errors only
END;

IF(@NotifyToEventLog = 1)
BEGIN
    SET @NotifyToEventLogIntVal = 2; -- Errors only
END;

IF(EXISTS(SELECT distinct OnSuccessAction FROM #AgentJobSteps EXCEPT SELECT TableValue FROM #StepDefaultValuesMapping WHERE CreationParamName = 'OnSuccessAction'))
BEGIN
    RAISERROR('Review contents of #AgentJobSteps table: there are invalid values for OnSuccessAction column',12,1);
    GOTO QuitWithRollback ;
END;

IF(EXISTS(SELECT distinct OnFailureAction FROM #AgentJobSteps EXCEPT SELECT TableValue FROM #StepDefaultValuesMapping WHERE CreationParamName = 'OnFailureAction'))
BEGIN
    RAISERROR('Review contents of #AgentJobSteps table: there are invalid values for OnFailureAction column',12,1);
    GOTO QuitWithRollback ;
END;

IF(EXISTS(SELECT distinct BehaviorOptions FROM #AgentJobSteps EXCEPT SELECT TableValue FROM #StepDefaultValuesMapping WHERE CreationParamName = 'BehaviorOptions'))
BEGIN
    RAISERROR('Review contents of #AgentJobSteps table: there are invalid values for BehaviorOptions column',12,1);
    GOTO QuitWithRollback ;
END;


IF(LEN(LTRIM(RTRIM(@AgentJobName))) = 0 )
BEGIN
    RAISERROR('Provide Agent Job Name',12,1);
    GOTO QuitWithRollback;
END;

IF(@NotifyByEmail = 0 AND @NotifyToEventLog = 0)
BEGIN
    RAISERROR('You should at least notify to one target when an error occurs',12,1);
    GOTO QuitWithRollback;
END;

SELECT @EventSessionName = CASE WHEN LEN(LTRIM(RTRIM(@EventSessionName))) = 0 THEN 'system_health' ELSE @EventSessionName END;

IF(OBJECT_ID('sys.dm_xe_sessions') IS NULL)
BEGIN
    RAISERROR('This version of SQL Server does not support extended events',12,1);
    GOTO QuitWithRollback ;
END;

IF(NOT EXISTS(SELECT 1 FROM sys.dm_xe_sessions WHERE name = @EventSessionName))
BEGIN
    RAISERROR('No Extended Event with name [%s] found on this server',12,1,@EventSessionName);
    GOTO QuitWithRollback ;
END;



-- Close condition: agent job already defined => stop (successfully)
SELECT @AgentJobId = job_id 
FROM msdb.dbo.sysjobs_view
WHERE name = @AgentJobName
;

IF(@AgentJobId IS NOT NULL)
BEGIN
    RAISERROR('Job [%s] already exists',0,1,@AgentJobName);
    RAISERROR('If necessary, drop the job and restart this script',0,1);
    GOTO EndSave;
END;

IF(@RunWithDebugOutput = 1)
BEGIN
    RAISERROR('Generating statements to create job steps',0,1) WITH NOWAIT;
END;


INSERT INTO #JobCreationCommands (
    Comments,Stmt
)   
select --s.*,Success_val.ActualValue,Failure_val.ActualValue,Behavior_val.ActualValue
    StepName ,
	'EXEC @ReturnCode =  msdb.dbo.sp_add_jobstep' + @LineFeed +
	'    @job_name               = ''' + @AgentJobName + ''',' + @LineFeed + 
	'    @step_name				 = ''' + StepName + ''',' + @LineFeed +
	'    @subsystem				 = ''' + StepLanguage + ''',' + @LineFeed +
	'    @command				 = ''' + REPLACE(StepText,'''','''''') + ''',' + @LineFeed +
	'    @cmdexec_success_code	 = ' + CONVERT(VARCHAR(10),ExpectedReturnedVal) + ',' + @LineFeed +
	'    @on_success_action      = ' + CONVERT(VARCHAR(10),Success_val.ActualValue) + ',' + @LineFeed +
	'    @on_success_step_id     = ' + CONVERT(VARCHAR(10),s.OnSuccessStepId) + ',' + @LineFeed +
	'    @on_fail_action         = ' + CONVERT(VARCHAR(10),Failure_val.ActualValue) + ',' + @LineFeed +
	'    @on_fail_step_id        = ' + CONVERT(VARCHAR(10),s.OnFailureStepId) + ',' + @LineFeed +
	'    @database_name          = ' + CASE WHEN DatabaseName IS NULL THEN 'NULL' ELSE '''' + DatabaseName + '''' END + ',' + @LineFeed + 
	'    @database_user_name     = ' + CASE WHEN DatabaseUserName IS NULL THEN 'NULL' ELSE '''' + DatabaseUserName + '''' END + ','  + @LineFeed + 
	'    @retry_attempts         = ' + CONVERT(VARCHAR(10), s.RetryAttemptsCount) + ',' + @LineFeed + 
	'    @retry_interval         = ' + CONVERT(VARCHAR(10), s.RetryInterval) + ','  + @LineFeed + 
	'    @output_file_name       = ' + CASE WHEN OutputFileName IS NULL THEN 'NULL' ELSE '''' + s.OutputFileName + '''' END + ',' + @LineFeed + 
	'    @flags                  = ' + CONVERT(VARCHAR(10),Behavior_Val.ActualValue) + ',' + @LineFeed +
	'    @proxy_id               = ' + CASE WHEN OutputFileName IS NULL THEN 'NULL' ELSE '''' + CONVERT(VARCHAR(10),s.ProxyId) + '''' END + ',' + @LineFeed + 
	'    @proxy_name             = ' + CASE WHEN OutputFileName IS NULL THEN 'NULL' ELSE '''' + s.ProxyName + '''' END + @LineFeed +
	';' + @LineFeed
From #AgentJobSteps s
INNER JOIN #StepDefaultValuesMapping Success_val 
ON  Success_val.CreationParamName = 'OnSuccessAction'
AND Success_val.TableValue        = s.OnSuccessAction
INNER JOIN #StepDefaultValuesMapping Failure_val 
ON  Failure_val.CreationParamName = 'OnFailureAction'
AND Failure_val.TableValue        = s.OnFailureAction
INNER JOIN #StepDefaultValuesMapping Behavior_val 
ON  Behavior_val.CreationParamName = 'BehaviorOptions'
AND Behavior_val.TableValue        = s.BehaviorOptions
;

IF(0 = (SELECT COUNT(*) FROM #JobCreationCommands WHERE Stmt IS NOT NULL))
BEGIN
    RAISERROR('Internal error or NULL user input. Please Check',12,1);
    GOTO QuitWithRollback;
END;

BEGIN TRANSACTION 

IF(NOT EXISTS (SELECT name FROM msdb.dbo.syscategories WHERE name=@AgentJobCategory AND category_class=1))
BEGIN

    IF(@RunWithDebugOutput = 1)
    BEGIN
        RAISERROR('Creating Job Category [%s] ',0,1,@AgentJobCategory);
    END;

    EXEC @ReturnCode = msdb.dbo.sp_add_category @class=N'JOB', @type=N'LOCAL', @name=@AgentJobCategory ;
    
    IF (@@ERROR <> 0 OR @ReturnCode <> 0) 
    BEGIN   
        GOTO QuitWithRollback;
    END;
END;

IF(@RunWithDebugOutput = 1)
BEGIN
    RAISERROR('Creating new job called "%s"',0,1,@AgentJobName);
END;

EXEC @ReturnCode =  msdb.dbo.sp_add_job 
                            @job_name                   = @AgentJobName, 
                            @enabled                    = @CreateJobEnabled, 
                            @notify_level_eventlog      = @NotifyToEventLogIntVal, 
                            @notify_level_email         = @NotifyByEmailIntVal, 
                            @notify_level_netsend       = 0, 
                            @notify_level_page          = 0, 
                            @delete_level               = 0, 
                            @description                = @AgentJobDescription, 
                            @category_name              = @AgentJobCategory, 
                            @owner_login_name           = N'sa', 
                            @notify_email_operator_name = @NotifyEmailOperator, 
                            @job_id                     = @AgentJobId OUTPUT
;
    
IF (@@ERROR <> 0 OR @ReturnCode <> 0)
BEGIN   
    GOTO QuitWithRollback;
END;

WHILE(1 = 1)
BEGIN
    SET @currentCommandId = NULL;
    
    SELECT
        @currentCommandId  = min(OrderId)
    FROM #JobCreationCommands
    ;
    
    IF(@currentCommandId IS NULL)
    BEGIN
        BREAK;
    END;
    
    SELECT  
        @currentStepComments    = Comments,
        @tsql                   = Stmt
    FROM #JobCreationCommands
    WHERE OrderId = @currentCommandId
    ;

    IF(NOT EXISTS (SELECT * FROM msdb.dbo.sysjobsteps WHERE job_id = @AgentJobId and step_id = @currentCommandId and step_name = @currentStepComments))
    BEGIN 
        IF(@RunWithDebugOutput = 1)
        BEGIN
            SET @LogMsg = 'Creating Job Step #' + CONVERT(VARCHAR(10),@currentCommandId);
            RAISERROR(@LogMsg,0,1);
			RAISERROR('----------------------- (TSQL Code) ----------------------------------',0,1);
			RAISERROR(@tsql,0,1);
			RAISERROR('------------------- (End of TSQL Code) -------------------------------',0,1);
        END;
        
        EXEC @ReturnCode = sp_executesql @tsql, N'@ReturnCode INT OUTPUT', @ReturnCode = @InnerReturnCode OUTPUT;

        IF (@@ERROR <> 0 OR @ReturnCode <> 0 OR @InnerReturnCode <> 0)
        BEGIN
            SET @LogMsg = 'Something went wront with creation statement for Job Step #' + CONVERT(VARCHAR(10),@currentCommandId);
            RAISERROR(@LogMsg,12,1);
            GOTO QuitWithRollback;
        END;
        
        IF(@RunWithDebugOutput = 1)
        BEGIN
            RAISERROR('Associating job and step...',0,1);
        END;
            
        EXEC @ReturnCode = msdb.dbo.sp_update_job @job_id = @AgentJobId, @start_step_id = @currentCommandId;

        IF (@@ERROR <> 0 OR @ReturnCode <> 0)
        BEGIN
            GOTO QuitWithRollback;
        END;
        
    END;
    ELSE
    BEGIN
        IF(@RunWithDebugOutput = 1)
        BEGIN
            SET @LogMsg = 'Ignoring Job Step #' + CONVERT(VARCHAR(10),@currentCommandId) + ' as it already exists';
            RAISERROR(@LogMsg,0,1);
        END;
    END;
    
    DELETE FROM #JobCreationCommands
    WHERE OrderId = @currentCommandId
    ;
END;

IF(@RunWithDebugOutput = 1)
BEGIN
    RAISERROR('Adding local server as job target...',0,1);
END;
    
EXEC @ReturnCode = msdb.dbo.sp_add_jobserver @job_id = @AgentJobId, @server_name = N'(local)'
IF (@@ERROR <> 0 OR @ReturnCode <> 0)
BEGIN
    GOTO QuitWithRollback;
END;


/*
    Here, the job is created, its steps too and #JobCreationCommands is empty.
    Now it's time to schedule this job!
    To do so, we'll generate commands as we did for job and store in into #JobCreationCommands
*/

IF(@RunWithDebugOutput = 1)
BEGIN
    RAISERROR('Defining job schedules...',0,1);
END;


with ActualValues (
	ScheduleId,ScheduleName,ScheduleFrequency,ScheduleFrequencyIntrVal,ScheduleFreqSubDayType,ScheduleFreqSubDayIntrval,
	ScheduleFrequencyRelativeIntrval,ScheduleFreqRecurrence,ScheduleActivationDateTime,ScheduleEndDateTime
)
AS (
	select 
		scheds.ScheduleId,
		scheds.ScheduleName,
		sdvmFreq.ActualValue as ScheduleFrequency,
		-- Schedule Freq Interval Value depends on Frequency value
		-- @See https://docs.microsoft.com/en-us/sql/relational-databases/system-stored-procedures/sp-add-jobschedule-transact-sql
		-- used String value for readability in preference to performance
		CASE scheds.ScheduleFrequency
			WHEN 'DAILY' THEN 
				scheds.ScheduleFreqIntrvalAsNumber
			WHEN 'WEEKLY' THEN	
				sdvmIntrvalDay.ActualValue
			WHEN 'MONTHLY' THEN 
				scheds.ScheduleFreqIntrvalAsNumber
			WHEN 'MONTHLY_INTRVAL' THEN  
				sdvmIntrvalRange.ActualValue
			ELSE 
				NULL -- frequency_interval is unused for ONCE, AT_AGENT_START, ON_IDLE_CONDITION    
		END as ScheduleFrequencyIntrVal,
		sdvmFreqSubDayT.ActualValue as ScheduleFreqSubDayType,
		scheds.ScheduleFreqSubDayIntrval,
		sdvmFreqRelIntrval.ActualValue as ScheduleFrequencyRelativeIntrval,
		scheds.ScheduleFreqRecurrence,
		scheds.ScheduleActivationDateTime,
		scheds.ScheduleEndDateTime
	from #JobSchedules scheds
	LEFT JOIN #StepDefaultValuesMapping sdvmFreq
	ON sdvmFreq.CreationParamName = 'ScheduleFrequency' AND sdvmFreq.TableValue = scheds.ScheduleFrequency
	LEFT JOIN #StepDefaultValuesMapping sdvmIntrvalDay
	ON sdvmIntrvalDay.CreationParamName = 'ScheduleFrequencyAsDay' AND sdvmIntrvalDay.TableValue = scheds.ScheduleFreqIntrvalAsDay
	LEFT JOIN #StepDefaultValuesMapping sdvmIntrvalRange
	ON sdvmIntrvalRange.CreationParamName = 'ScheduleFreqIntrvalAsRange' AND sdvmIntrvalRange.TableValue = scheds.ScheduleFreqIntrvalAsRange
	LEFT JOIN #StepDefaultValuesMapping sdvmFreqSubDayT
	ON sdvmFreqSubDayT.CreationParamName = 'ScheduleFreqSubDayType' AND sdvmFreqSubDayT.TableValue = scheds.ScheduleFreqSubDayType
	LEFT JOIN #StepDefaultValuesMapping sdvmFreqRelIntrval
	ON sdvmFreqRelIntrval.CreationParamName = 'ScheduleFreqRelativeIntrval' AND sdvmFreqRelIntrval.TableValue = scheds.ScheduleFreqRelativeIntrval
)
INSERT INTO #JobCreationCommands (
    Comments,Stmt
)   
SELECT
	ScheduleName,
	'EXEC msdb.dbo.sp_add_jobschedule' + @LineFeed + 
	'        @job_name                = ''' + @AgentJobName + ''',' + @LineFeed +
	'        @name                    = ''' + ScheduleName + ''',' + @LineFeed +
	'        @freq_type               = ' + CONVERT(VARCHAR(10),ScheduleFrequency) + ',' + @LineFeed +
	'        @freq_interval           = ' + CONVERT(VARCHAR(10),ScheduleFrequencyIntrVal) + ',' + @LineFeed +
	'        @freq_subday_type        = ' + CASE WHEN ScheduleFreqSubDayType IS NULL THEN 'NULL' ELSE CONVERT(VARCHAR(10),ScheduleFreqSubDayType) END + ',' + @LineFeed +
	'        @freq_subday_interval    = ' + CONVERT(VARCHAR(10),ScheduleFreqSubDayIntrval) + ',' + @LineFeed +
	'        @freq_relative_interval  = ' + CASE WHEN ScheduleFrequencyRelativeIntrval IS NULL THEN 'NULL' ELSE CONVERT(VARCHAR(10),ScheduleFrequencyRelativeIntrval) END + ',' + @LineFeed +
	'        @freq_recurrence_factor  = ' + CONVERT(VARCHAR(10),ScheduleFreqRecurrence) + ',' + @LineFeed + 
	'        @active_start_date       = ' + CONVERT(VARCHAR(16),ScheduleActivationDateTime,112) + ',' + @LineFeed +
	'        @active_end_date         = ' + CASE 
	                                           WHEN ScheduleEndDateTime IS NULL THEN 'NULL' 
	                                           ELSE CONVERT(VARCHAR(16),ScheduleEndDateTime,112) 
											END + ',' + @LineFeed +
	'        @active_start_time       = ' + REPLACE(CONVERT(VARCHAR(16),ScheduleActivationDateTime,108),':','') + ',' + @LineFeed +
	'        @active_end_time         = ' + CASE 
	                                           WHEN ScheduleEndDateTime IS NULL THEN 'NULL' 
	                                           ELSE REPLACE(CONVERT(VARCHAR(16),ScheduleEndDateTime,108),':','')
											END + ',' + @LineFeed +
	'        @schedule_id             = @ScheduleId OUTPUT' + @LineFeed +
	';'  

FROM ActualValues
ORDER BY ScheduleId
;

WHILE(1 = 1)
BEGIN
    SET @currentCommandId = NULL;
    
    SELECT
        @currentCommandId  = min(OrderId)
    FROM #JobCreationCommands
    ;
    
    IF(@currentCommandId IS NULL)
    BEGIN
        BREAK;
    END;
    
    SELECT  
        @currentStepComments    = Comments, -- equals schedule's name
        @tsql                   = Stmt
    FROM #JobCreationCommands
    WHERE OrderId = @currentCommandId
    ;
    
    
    IF(NOT EXISTS (SELECT * FROM msdb.dbo.sysschedules WHERE name = @currentStepComments))
    BEGIN 
        IF(@RunWithDebugOutput = 1)
        BEGIN
            SET @LogMsg = 'Creating Schedule with name [' + @currentStepComments + ']';
            RAISERROR(@LogMsg,0,1);

			RAISERROR('----------------------- (TSQL Code) ----------------------------------',0,1);
			RAISERROR(@tsql,0,1);
			RAISERROR('------------------- (End of TSQL Code) -------------------------------',0,1);
        END;
        
        EXEC @ReturnCode = sp_executesql @tsql, N'@ExecRet INT OUTPUT,@ScheduleId INT OUTPUT', @ExecRet = @InnerReturnCode OUTPUT, @ScheduleId = @JobScheduleId;

        IF (@@ERROR <> 0 OR @ReturnCode <> 0 OR @InnerReturnCode <> 0)
        BEGIN
            SET @LogMsg = 'Something went wront with creation statement for schedule with name [' + @currentStepComments + ']';
            RAISERROR(@LogMsg,12,1);
            GOTO QuitWithRollback;
        END;
    END;
    ELSE
    BEGIN
        IF(@RunWithDebugOutput = 1)
        BEGIN
            SET @LogMsg = 'Ignoring creation of schedule with name [' + @currentStepComments + ']' + ' as it already exists';
            RAISERROR(@LogMsg,0,1);
        END;
    END;
    
    DELETE FROM #JobCreationCommands
    WHERE OrderId = @currentCommandId
    ;
END;



COMMIT TRANSACTION;
GOTO EndSave;



--
-- Script Ending
-- ============================================


QuitWithRollback:
    IF (@@TRANCOUNT > 0)
    BEGIN 
        ROLLBACK TRANSACTION
    END;
    
EndSave:

    IF(OBJECT_ID('tempdb..#AgentJobSteps') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #AgentJobSteps;';
    END;

    IF(OBJECT_ID('tempdb..#JobCreationCommands') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #JobCreationCommands;';
    END;
    
    IF(OBJECT_ID('tempdb..#StepDefaultValuesMapping') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #StepDefaultValuesMapping;';
    END;    
    
    IF(OBJECT_ID('tempdb..#JobSchedules') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #JobSchedules;';
    END;

GO

