/*requires Schema.Utils.sql*/

DECLARE @ProcedureSchema NVARCHAR(256);
DECLARE @ProcedureName   NVARCHAR(256);

SET @ProcedureSchema = 'Utils' ;
SET @ProcedureName = 'ManageExtendedProperty' ;

RAISERROR('-----------------------------------------------------------------------------------------------------------------',0,1);
RAISERROR('PROCEDURE [%s].[%s]',0,1,@ProcedureSchema,@ProcedureName);

IF  NOT EXISTS (SELECT 1 FROM sys.objects WHERE object_id = OBJECT_ID(N'[' + @ProcedureSchema + N'].[' + @ProcedureName +  N']') AND type in (N'P'))
BEGIN
    BEGIN TRY
        EXECUTE ('CREATE Procedure [' + @ProcedureSchema + '].[' + @ProcedureName +  '] ( ' +
                ' @ServerName    varchar(512), ' +
                ' @DbName    varchar(50) ' +
                ') ' +
                'AS ' +
                'BEGIN ' +
                '   SELECT ''Not implemented'' ' +
                'END')
    END TRY
    BEGIN CATCH
        PRINT '   Error while trying to create procedure'
        RETURN
    END CATCH

    PRINT '   PROCEDURE created.'
END
GO

ALTER PROCEDURE [Utils].[ManageExtendedProperty] (
    @DatabaseName                   VARCHAR(256)   = NULL,
    @PropertyName                   VARCHAR(256)   = 'MS_Description',
    @PropertyValue                  VARCHAR(1024),
    @LevelType_0                    VARCHAR(128)   = 'SCHEMA', -- Valid inputs are ASSEMBLY, CONTRACT, EVENT NOTIFICATION, FILEGROUP, MESSAGE TYPE, PARTITION FUNCTION, PARTITION SCHEME, PLAN GUIDE, REMOTE SERVICE BINDING, ROUTE, SCHEMA, SERVICE, USER, TRIGGER, TYPE, and NULL. 
    @LevelName_0                    VARCHAR(128),
    @LevelType_1                    VARCHAR(128)   = NULL, -- Valid inputs are AGGREGATE, DEFAULT, FUNCTION, LOGICAL FILE NAME, PROCEDURE, QUEUE, RULE, SYNONYM, TABLE, TABLE_TYPE, TYPE, VIEW, XML SCHEMA COLLECTION, and NULL. 
    @LevelName_1                    VARCHAR(128)   = NULL,
    @LevelType_2                    VARCHAR(128)   = NULL, -- Valid inputs are COLUMN, CONSTRAINT, EVENT NOTIFICATION, INDEX, PARAMETER, TRIGGER, and NULL. 
    @LevelName_2                    VARCHAR(128)   = NULL,
    @OperationMode                  VARCHAR(32)    = 'Overwrite', -- Valid inputs are 'OVERWRITE','APPEND','REMOVE', 'PREPEND'
    @_NoChecks                      BIT            = 0,
    @Debug                          BIT            = 0
)
AS
/*
  ===================================================================================
    @See https://docs.microsoft.com/en-us/sql/relational-databases/system-stored-procedures/sp-updateextendedproperty-transact-sql 

    DESCRIPTION:

    PARAMETERS:

    REQUIREMENTS:

    EXAMPLE USAGE :
    
        -- Schema property
        EXEC [Utils].[ManageExtendedProperty] 
                    @PropertyValue = 'This is a utility tools schema', 
                    @LevelType_0   = 'SCHEMA', 
                    @LevelName_0   = 'Utils'
        ;

        -- procedure property
        EXEC [Utils].[ManageExtendedProperty] 
                    @PropertyValue	= 'Procedure for extended properties definition and edition', 
                    @LevelType_0	= 'SCHEMA', 
                    @LevelName_0	= 'Utils',
                    @LevelType_1	= 'PROCEDURE' , 
                    @LevelName_1	= 'ManageExtendedProperty'
        ;

        -- table property
        EXEC [Utils].[ManageExtendedProperty] 
                    @PropertyValue	= 'Contains all references about schema objects per database', 
                    @LevelType_0	= 'SCHEMA', 
                    @LevelName_0	= 'Docs',
                    @LevelType_1	= 'TABLE' , 
                    @LevelName_1	= 'SchemaObjectDictionary'
        ;

        -- table column property
        EXEC [Utils].[ManageExtendedProperty] 
                    @PropertyValue	= 'Name of database in which run procedure', 
                    @LevelType_0	= 'SCHEMA', 
                    @LevelName_0	= 'Docs',
                    @LevelType_1	= 'TABLE' , 
                    @LevelName_1	= 'SchemaObjectDictionary',
                    @LevelType_2    = 'COLUMN',
                    @LevelName_2    = 'DatabaseName'
        ;

        -- Procedure parameter 
        EXEC [Utils].[ManageExtendedProperty] 
                    @PropertyValue	= 'Name of database in which run procedure', 
                    @LevelType_0	= 'SCHEMA', 
                    @LevelName_0	= 'Utils',
                    @LevelType_1	= 'PROCEDURE' , 
                    @LevelName_1	= 'ManageExtendedProperty',
                    @LevelType_2    = 'PARAMETER',
                    @LevelName_2    = '@DatabaseName', -- DO NOT FORGET THE AROBAS '@'  !!!
                    @Debug = 1
        ;

        -- Check out:

        SELECT * FROM sys.extended_properties WHERE [name] = N'MS_Description'

  ===================================================================================
*/
BEGIN
    SET NOCOUNT ON;
    DECLARE @tsql               nvarchar(max);
    DECLARE @TmpStr             nvarchar(max);
    DECLARE @TmpInt             BIGINT;
    DECLARE @LineFeed           CHAR(2);
    DECLARE @isSchemaObject     BIT ;
    DECLARE @isAddOperation     BIT ;
    DECLARE @MajorIdText        VARCHAR(2048);
    DECLARE @MinorIdText        VARCHAR(2048);
    DECLARE @FullLevelType      VARCHAR(2048);
    DECLARE @MSSQLVersionMajor  SMALLINT;
	DECLARE @ExecRet            INT;
    DECLARE @CanOperate         BIT;

    SELECT
        @CanOperate         = 0,
        @MSSQLVersionMajor  = CONVERT(INT,@@MICROSOFTVERSION / 0x01000000),
        @tsql               = '',
        @LineFeed           = CHAR(13) + CHAR(10),
        @isSchemaObject     = 0,
        @isAddOperation     = 1,
        @OperationMode      = UPPER(@OperationMode),
        @OperationMode      = CASE WHEN LEN(LTRIM(RTRIM(@OperationMode))) = 0 THEN NULL ELSE @OperationMode END,
        @DatabaseName       = CASE WHEN @DatabaseName IS NULL OR LEN(LTRIM(RTRIM(@DatabaseName))) = 0 THEN DB_NAME() ELSE @DatabaseName  END,
        -- input adjustments:
        @LevelType_1        = CASE @LevelType_1 
                                WHEN 'USER_TABLE' THEN 'TABLE' 
                                WHEN 'TABLE TYPE' THEN 'TYPE' -- Should be 'TABLE_TYPE' but won't work (tested on SQL Server 2012)
                                ELSE @LevelType_1 END,
        @FullLevelType      = CASE 
                                WHEN @LevelType_0 IS NULL THEN '' 
                                ELSE @LevelType_0 + CASE    
                                                        WHEN @LevelType_1 IS NULL THEN '' 
                                                        ELSE '.' + @LevelType_1 + CASE 
                                                                                    WHEN @LevelType_2 IS NULL THEN '' 
                                                                                    ELSE '.' + @LevelType_2 
                                                                                  END 
                                                    END 
                              END
    ;

    if (@Debug = 1)
    BEGIN
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
        RAISERROR('-- Now running [Utils].[ManageExtendedProperty] stored procedure.',0,1);
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
    END;

    IF(@OperationMode NOT IN ('OVERWRITE','APPEND','PREPEND','REMOVE'))
    BEGIN
        RAISERROR('Invalid operation Mode [%s]',12,1,@OperationMode);
        RETURN;
    END;
    
    IF((@LevelType_1 IS NOT NULL AND @LevelType_0 IS NULL) OR (@LevelType_2 IS NOT NULL AND @LevelType_1 IS NULL))
    BEGIN
        RAISERROR('Parameter problem with @Level_Type_* values',12,1);
        RETURN;
    END;
    
    IF(DB_ID(@DatabaseName) IS NULL)
    BEGIN
        RAISERROR('Database [%s] not found on server',12,1,@DatabaseName);
        RETURN;
    END;
    
    IF(OBJECT_ID('tempdb..#ValidParamAssociations') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #ValidParamAssociations;';
    END;
    
    CREATE TABLE #ValidParamAssociations (
        AssociationName         VARCHAR(256) NOT NULL PRIMARY KEY CLUSTERED,
        Level0                  VARCHAR(256),
        Level1                  VARCHAR(256),
        Level2                  VARCHAR(256),
        MinVersionMajor         VARCHAR(8),
        MaxVersionMajor         VARCHAR(8),
        hasImplementedHandler   BIT         NOT NULL,
        FullPathValue           as 
                                    CASE 
                                        WHEN Level0 is NULL THEN '' 
                                        ELSE Level0 +
                                            CASE 
                                                WHEN Level1 IS NULL THEN ''
                                                ELSE '.' + Level1 +
                                                    CASE 
                                                        WHEN Level2 IS NULL THEN ''
                                                        ELSE '.' + Level2
                                                    END
                                            END
                                    END PERSISTED
    );
    
    INSERT INTO #ValidParamAssociations (
        AssociationName, Level0, Level1, Level2, MinVersionMajor, MaxVersionMajor, hasImplementedHandler
    )
    VALUES
        -- Database configuration related
        ('Database',                                NULL, NULL, NULL,                           '8',  NULL, 0),
        ('FileGroup',                               'fileGroup', NULL, NULL,                    '9', NULL,0),
        ('fileGroup.Logical file Name',             'fileGroup', 'Logical file Name', NULL,     '9', NULL,0),
        ('Asymmetric Key',                          'Asymmetric Key', NULL, NULL,               '10', NULL, 0),
        ('Certificate',                             'Certificate', NULL, NULL,                  '10', NULL,0),
        ('Plan guide',                              'plan guide', NULL, NULL,                   '10', NULL,0),
        
        -- Service Broker related
        ('Service Broker Contract',                 'Contract', NULL, NULL,                     '9',  NULL,0),
        ('Service Broker Queue',                    'schema', 'Queue', NULL,                    '9', NULL, 0),
        ('Service Broker Event Notification (Queue)','schema', 'Queue','Event Notification',     '9', NULL,0 ),
        ('Service Broker Remote Service Binding',   'Remote Service Binding', NULL, NULL,       '9', NULL,0),
        ('Service Broker Route',                    'route', NULL, NULL,                        '9', NULL, 0),
        ('Service Broker Service',                  'Service', NULL, NULL,                      '9', NULL, 0),
        ('Service Broker Schema Service',           'schema','Service', NULL,                   '9', NULL, 0),
        ('Service Broker Event Notification',       'Event Notification', NULL, NULL,           '9', NULL,0), /* https://technet.microsoft.com/en-us/library/ms182602(v=sql.105).aspx */
        ('Service Broker Message type',             'Message type', NULL, NULL,                 '9', NULL,0), /* https://docs.microsoft.com/en-us/sql/t-sql/statements/create-message-type-transact-sql */
        
        -- Database Objects
        ('Schema',                                  'Schema', NULL, NULL,                       '9', NULL, 0),
        ('Database Trigger',                        'trigger', NULL, NULL,                      '9', NULL,0),
        ('Symmetric Key',                           'Symmetric Key', NULL, NULL,                '9', NULL,0),
        ('Assembly',                                'assembly', NULL, NULL,                     '9',  NULL,0),
        ('Partition Function',                      'partition Function', NULL, NULL,           '9', NULL,0),
        ('Partition Scheme',                        'partition Scheme', NULL, NULL,             '9', NULL,0),
        
        -- Schema objects
        ('Schema Function',                         'schema', 'function', NULL,                 '9', NULL,1), -- OK
        ('Schema Function Parameter',               'Schema', 'Function', 'Parameter',          '9', NULL,1), -- OK
        ('Schema Procedure',                        'schema', 'Procedure', NULL,                '9', NULL,1), -- OK 
        ('Schema Procedure Parameter',              'schema', 'Procedure', 'Parameter',         '9', NULL,1), -- OK 
        ('Schema Rule',                             'schema', 'Rule', NULL,                     '9', NULL, 0), 
        ('Schema Synonym',                          'Schema', 'Synonym', NULL,                  '9', NULL, 0),
        ('Schema Table',                            'Schema', 'Table', NULL,                    '9', NULL,1), -- OK
        ('Schema Table Column',                     'Schema', 'Table', 'Column',                '9', NULL, 1), -- OK
        ('Schema Table Constraint',                 'Schema', 'Table', 'Constraint',            '9', NULL,0),
        ('Schema Table Index',                      'Schema', 'Table', 'Index',                 '9', NULL,0),
        ('Schema Table Trigger',                    'Schema', 'Table', 'Trigger',               '9', NULL,0),
        ('Schema Type',                             'Schema', 'Type', NULL,                     '9', NULL,1), -- OK
        ('Schema View',                             'Schema', 'View', NULL,                     '9', NULL,1), -- OK 
        ('Schema View column',                      'Schema', 'View', 'column',                 '9', NULL,1), -- OK 
        ('Schema View index',                       'Schema', 'View', 'index',                  '9', NULL,0),
        ('Schema View Trigger',                     'Schema', 'View', 'Trigger',                '9', NULL,0),
        ('Schema Aggregate Function',               'schema', 'aggregate', NULL,                '9',  NULL,0), /* https://docs.microsoft.com/en-us/sql/t-sql/statements/create-aggregate-transact-sql */
        ('Schema Table Type',                       'schema','table_type',NULL,                 '12',NULL,0), -- Not working
        
        -- ???
        ('Schema.Default',                          'schema', 'Default', NULL,                  '9',  NULL,0),
        ('Synonym',                                 'Synonym', NULL, NULL,                      '10', NULL,0),
        ('XML Schema Collection',                   'Schema', 'XML Schema Collection', NULL,    '9', NULL,0)
    ;
        
    IF(
        NOT EXISTS(
            SELECT 1 
            FROM #ValidParamAssociations 
            WHERE FullPathValue = @FullLevelType 
            AND (MinVersionMajor IS NULL OR MinVersionMajor <= @MSSQLVersionMajor) 
            AND (MaxVersionMajor IS NULL OR MaxVersionMajor >= @MSSQLVersionMajor)
        )
    )
    BEGIN
        RAISERROR('Invalid parameter association [%s]',12,1,@FullLevelType);
        RETURN;
    END;
    
    IF(@_NoChecks = 0 AND NOT EXISTS(SELECT 1 FROM #ValidParamAssociations WHERE FullPathValue = @FullLevelType AND hasImplementedHandler = 1))
    BEGIN
        RAISERROR('Handler for parameter association [%s] not implemented yet.',12,1,@FullLevelType);
        RETURN;
    END
    
    -- nomore needed
    IF(OBJECT_ID('tempdb..#ValidParamAssociations') IS NOT NULL)
    BEGIN
        EXEC sp_executesql N'DROP TABLE #ValidParamAssociations;';
    END;
    
    IF(@_NoChecks = 0)
    BEGIN
        IF(@LevelType_0 = 'SCHEMA')
        BEGIN
            -- check schema exists
            SET @tsql = 'USE ' + QUOTENAME(@DatabaseName) + ';' + @LineFeed +
                        'SELECT @ObjId = SCHEMA_ID(@SchemaName);'
                        
            EXEC sp_executesql @tsql , N'@ObjId BIGINT OUTPUT,@SchemaName VARCHAR(256)', @ObjId = @TmpInt OUTPUT , @SchemaName = @LevelName_0 ;
            
            IF(@TmpInt IS NULL)
            BEGIN
                RAISERROR('Schema with name [%s] not found in database',12,1,@LevelName_0);
                RETURN;
            END;
            
            SET @isSchemaObject = 1;
        END;
        ELSE
        BEGIN
            RAISERROR('level type 0 value [%s] not handled at the moment',12,1);
            RETURN;
        END;
    
        IF(@isSchemaObject = 1)
        BEGIN
            SET @MajorIdText = 'SCHEMA_ID(''' + @LevelName_0 + ''')';
            SET @MinorIdText = '0';
            
            IF(@LevelType_1 IS NOT NULL AND @LevelType_1 NOT IN ('TYPE','TABLE_TYPE'))
            BEGIN
                -- check object exists 
                SET @tsql = 'USE ' + QUOTENAME(@DatabaseName) + ';' + @LineFeed +
                            'SELECT @ObjId = OBJECT_ID(@ObjName);'
                
                SET @TmpStr = QUOTENAME(@LevelName_0) + '.' + QUOTENAME(@LevelName_1) ;
                
                EXEC sp_executesql @tsql , N'@ObjId BIGINT OUTPUT,@ObjName VARCHAR(1024)', @ObjId = @TmpInt OUTPUT , @ObjName = @TmpStr ;
                
                IF(@TmpInt IS NULL)
                BEGIN
                    RAISERROR('Object [%s] not found in database',12,1,@TmpStr);
                    RETURN;
                END;
                
                SET @MajorIdText = 'OBJECT_ID(''' + QUOTENAME(@LevelName_0) + '.' + QUOTENAME(@LevelName_1) + ''')';
            END;
            ELSE IF(@LevelType_1 IS NOT NULL AND @LevelType_1 IN ('TYPE','TABLE_TYPE'))
            BEGIN
                -- check type exists 
                SET @tsql = 'USE ' + QUOTENAME(@DatabaseName) + ';' + @LineFeed +
                            'select @Cnt = count(1) From INFORMATION_SCHEMA.DOMAINS WHERE DOMAIN_SCHEMA = @ObjSchema AND DOMAIN_NAME = @ObjName;'
                            ;
                
                EXEC sp_executesql 
                        @tsql , 
                        N'@Cnt INT OUTPUT,@ObjSchema VARCHAR(256),@ObjName VARCHAR(256)', 
                        @Cnt        = @TmpInt OUTPUT , 
                        @ObjSchema  = @LevelName_0, 
                        @ObjName    = @LevelName_1  
                ;
                
                IF(@TmpInt IS NULL)
                BEGIN
                    RAISERROR('Object [%s] not found in database',12,1,@TmpStr);
                    RETURN;
                END;      
                
                SET @MajorIdText = '(select user_type_id From sys.types where schema_id = SCHEMA_ID('''+ @LevelName_0 + ''') and name = ''' + @LevelName_1  + ''')';                
            END;
            
            IF(@LevelType_2 IS NOT NULL) -- then @LevelType_1 is not null neither
            BEGIN
                IF(     (@LevelType_1 IN ('TABLE','VIEW') AND @LevelType_2 <> 'COLUMN')
                    OR  (@LevelType_1 IN ('PROCEDURE','FUNCTION') AND @LevelType_2 <> 'PARAMETER')
                )
                BEGIN
                    RAISERROR('Cannot associate level type 1 [%s] with level type 2 [%s]',12,1,@LevelType_1,@LevelType_2);
                    RETURN;
                END;
                ELSE IF (@LevelType_1 IN ('TABLE','VIEW') AND @LevelType_2 = 'COLUMN')
                BEGIN
                    -- TODO: check existency for @LevelName_2 column 
                    SET @LevelType_2 = @LevelType_2;
                    -- minor_id value is the column id for table/view objects
                    SET @MinorIdText = '(SELECT [column_id] FROM SYS.COLUMNS WHERE [name] = ''' + @LevelName_2 + ''' AND object_id = ' + @MajorIdText + ')' ;
                END;
                ELSE IF (@LevelType_1 IN ('PROCEDURE','FUNCTION') AND @LevelType_2 = 'PARAMETER')
                BEGIN
                    -- TODO: check existency for @LevelName_2 parameter 
                    
                    -- minor_id value is the parameter id for programmable objects
                    SET @LevelType_2 = @LevelType_2;
                    SET @MinorIdText = '(select parameter_id From sys.parameters where [name] = ''' + @LevelName_2 + ''' AND object_id = ' + @MajorIdText + ')' ;
                END;
                ELSE
                BEGIN
                    RAISERROR('Unhandled level type 1 [%s] with level type 2 [%s]',12,1,@LevelType_1,@LevelType_2);
                    RETURN;
                END;
                
                -- TODO: review when other values for @LevelType_1 will be available
                --SET @MinorIdText = '(SELECT [column_id] FROM SYS.COLUMNS WHERE [name] = ''' + @LevelName_2 + ''' AND object_id = ' + @MajorIdText + ')' ;
            END;
            
        END;
        ELSE
        BEGIN
            RAISERROR('should not happen at first implementation',12,1);
            RETURN;
        END;
    END;
    
    -- Getting back existing value (if any)
    IF(@_NoChecks = 0)
    BEGIN
        SET @tsql = 'USE ' + QUOTENAME(@DatabaseName) + ';' + @LineFeed +
                    'SELECT @val = CONVERT(VARCHAR(MAX),value)'+ @LineFeed +
                    'FROM sys.extended_properties' + @LineFeed +
                    'WHERE major_id = ' + @MajorIdText + @LineFeed +
                    'AND   minor_id = ' + @MinorIdText + @LineFeed +
                    'AND  [name]    = N''' + @PropertyName + + '''' + @LineFeed + 
                    ';' + @LineFeed 
                    ;
    END;
    ELSE 
    BEGIN
        SET @tsql = 'USE ' + QUOTENAME(@DatabaseName) + ';' + @LineFeed +
                    'DECLARE @PropertyValTbl TABLE(PropertyValue SQL_VARIANT);' + @LineFeed +
                    'SELECT @val = CONVERT(VARCHAR(MAX),value)'+ @LineFeed +
                    'FROM fn_listextendedproperty (' + @LineFeed +
                    '    N''' + @PropertyName + ''', ' + @LineFeed + 
                    '    ' + CASE WHEN @LevelType_0 IS NULL THEN 'NULL' ELSE '''' + @LevelType_0 + '''' END + ',' + @LineFeed +
                    '    ' + CASE WHEN @LevelName_0 IS NULL THEN 'NULL' ELSE '''' + @LevelName_0 + '''' END + ',' + @LineFeed +
                    '    ' + CASE WHEN @LevelType_1 IS NULL THEN 'NULL' ELSE '''' + @LevelType_1 + '''' END + ',' + @LineFeed +
                    '    ' + CASE WHEN @LevelName_1 IS NULL THEN 'NULL' ELSE '''' + @LevelName_1 + '''' END + ',' + @LineFeed +
                    '    ' + CASE WHEN @LevelType_2 IS NULL THEN 'NULL' ELSE '''' + @LevelType_2 + '''' END + ',' + @LineFeed +
                    '    ' + CASE WHEN @LevelName_2 IS NULL THEN 'NULL' ELSE '''' + @LevelName_2 + '''' END  + @LineFeed +
                    ');'
                    ;
    END;
    
    IF(@Debug = 1)
    BEGIN
        RAISERROR('Now running following statement:',0,1);
        RAISERROR('=====================================================',0,1);
        RAISERROR(@tsql,0,1);
        RAISERROR('=====================================================',0,1);
    END;
    
    SET @TmpStr = NULL;
    
    EXEC @ExecRet = sp_executesql @tsql , N'@val VARCHAR(MAX) OUTPUT', @val = @TmpStr OUTPUT;
    
    IF(@ExecRet <> 0 OR @TmpStr IS NULL)
    BEGIN 
        SET @isAddOperation = 1;
    END;
    ELSE --IF(@ExecRet = 0 AND @TmpStr IS NOT NULL)
    BEGIN
        SET @isAddOperation = 0; -- already a value defined
    END;
    
    IF(@isAddOperation = 0 AND @OperationMode = 'REMOVE')
    BEGIN
        IF(@Debug = 1)
        BEGIN
            RAISERROR('Removing extended property',0,1);
        END;
        SET @tsql = 'USE ' + QUOTENAME(@DatabaseName) + ';' + @LineFeed +
                    'EXEC @TmpInt = sp_dropextendedproperty ' + @LineFeed +
                    '        @name       = ''' + @PropertyName + ''', ' + @LineFeed +
                    '        @level0type = ' + CASE WHEN @LevelType_0 IS NULL THEN 'NULL' ELSE '''' + @LevelType_0 + '''' END + ',' + @LineFeed +
                    '        @level0name = ' + CASE WHEN @LevelName_0 IS NULL THEN 'NULL' ELSE '''' + @LevelName_0 + '''' END + ',' + @LineFeed +
                    '        @level1type = ' + CASE WHEN @LevelType_1 IS NULL THEN 'NULL' ELSE '''' + @LevelType_1 + '''' END + ',' + @LineFeed +
                    '        @level1name = ' + CASE WHEN @LevelName_1 IS NULL THEN 'NULL' ELSE '''' + @LevelName_1 + '''' END + ',' + @LineFeed +
                    '        @level2type = ' + CASE WHEN @LevelType_2 IS NULL THEN 'NULL' ELSE '''' + @LevelType_2 + '''' END + ',' + @LineFeed +
                    '        @level2name = ' + CASE WHEN @LevelName_2 IS NULL THEN 'NULL' ELSE '''' + @LevelName_2 + '''' END  + @LineFeed +
                    ';'
                    ;
        SET @CanOperate = 1;
    END;
    
    -- some adjustments to editing based on @OperationMode
    
    IF(@OperationMode = 'APPEND')
    BEGIN
        SET @PropertyValue = CASE WHEN @TmpStr IS NULL THEN @PropertyValue ELSE @TmpStr + @LineFeed + @PropertyValue END;
    END;
    ELSE IF (@OperationMode = 'PREPEND')
    BEGIN
        SET @PropertyValue = CASE WHEN @TmpStr IS NULL THEN @PropertyValue ELSE @PropertyValue + @LineFeed + @TmpStr  END;
    END;
    
    -- Now, let's 
    
    IF(@CanOperate = 0) -- No operation already defined
    BEGIN 
        IF(@isAddOperation = 1)
        BEGIN
            IF(@Debug = 1)
            BEGIN
                RAISERROR('New Extended Property will be defined',0,1);
            END;
            SET @tsql = 'USE ' + QUOTENAME(@DatabaseName) + ';' + @LineFeed +
                        'EXEC @TmpInt = sp_addextendedproperty ' + @LineFeed +
                        '        @name       = ''' + @PropertyName + ''', ' + @LineFeed +
                        '        @value      = ' + CASE WHEN @PropertyValue IS NULL THEN 'NULL' ELSE '''' + @PropertyValue + '''' END + ',' + @LineFeed +
                        '        @level0type = ' + CASE WHEN @LevelType_0 IS NULL THEN 'NULL' ELSE '''' + @LevelType_0 + '''' END + ',' + @LineFeed +
                        '        @level0name = ' + CASE WHEN @LevelName_0 IS NULL THEN 'NULL' ELSE '''' + @LevelName_0 + '''' END + ',' + @LineFeed +
                        '        @level1type = ' + CASE WHEN @LevelType_1 IS NULL THEN 'NULL' ELSE '''' + @LevelType_1 + '''' END + ',' + @LineFeed +
                        '        @level1name = ' + CASE WHEN @LevelName_1 IS NULL THEN 'NULL' ELSE '''' + @LevelName_1 + '''' END + ',' + @LineFeed +
                        '        @level2type = ' + CASE WHEN @LevelType_2 IS NULL THEN 'NULL' ELSE '''' + @LevelType_2 + '''' END + ',' + @LineFeed +
                        '        @level2name = ' + CASE WHEN @LevelName_2 IS NULL THEN 'NULL' ELSE '''' + @LevelName_2 + '''' END  + @LineFeed +
                        ';'
                        ;
        END;
        ELSE    
        BEGIN
            IF(@Debug = 1)
            BEGIN
                RAISERROR('Existing Extended Property will be updated',0,1);
            END;
            SET @tsql = 'USE ' + QUOTENAME(@DatabaseName) + ';' + @LineFeed +
                        'EXEC @TmpInt = sp_updateextendedproperty ' + @LineFeed +
                        '        @name       = ''' + @PropertyName + ''', ' + @LineFeed +
                        '        @value      = ' + CASE WHEN @PropertyValue IS NULL THEN 'NULL' ELSE '''' + @PropertyValue + '''' END + ',' + @LineFeed +
                        '        @level0type = ' + CASE WHEN @LevelType_0 IS NULL THEN 'NULL' ELSE '''' + @LevelType_0 + '''' END + ',' + @LineFeed +
                        '        @level0name = ' + CASE WHEN @LevelName_0 IS NULL THEN 'NULL' ELSE '''' + @LevelName_0 + '''' END + ',' + @LineFeed +
                        '        @level1type = ' + CASE WHEN @LevelType_1 IS NULL THEN 'NULL' ELSE '''' + @LevelType_1 + '''' END + ',' + @LineFeed +
                        '        @level1name = ' + CASE WHEN @LevelName_1 IS NULL THEN 'NULL' ELSE '''' + @LevelName_1 + '''' END + ',' + @LineFeed +
                        '        @level2type = ' + CASE WHEN @LevelType_2 IS NULL THEN 'NULL' ELSE '''' + @LevelType_2 + '''' END + ',' + @LineFeed +
                        '        @level2name = ' + CASE WHEN @LevelName_2 IS NULL THEN 'NULL' ELSE '''' + @LevelName_2 + '''' END + @LineFeed +
                        ';'
                        ;
        END;
        SET @CanOperate = 1;
    END;
    
    -- Now, let's run @sql defined above !
    
    IF(@CanOperate = 1)
    BEGIN
        IF(@Debug = 1)
        BEGIN
            RAISERROR('Now running following statement:',0,1);
            RAISERROR('=====================================================',0,1);
            RAISERROR(@tsql,0,1);
            RAISERROR('=====================================================',0,1);
        END;
        
        EXEC sp_executesql @tsql , N'@TmpInt BIGINT OUTPUT', @TmpInt = @TmpInt OUTPUT ;
        
        IF(@TmpInt <> 0)
        BEGIN
            SET @TmpStr = ERROR_MESSAGE();
            RAISERROR(@TmpStr,12,1);
            RETURN;
        END;
    END;
    ELSE 
    BEGIN
        IF(@Debug = 1)
        BEGIN
            RAISERROR('Due to previously logged reason, no operation has been made on extended properties',0,1);
        END;
    END;
    if (@Debug = 1)
    BEGIN
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
        RAISERROR('-- Execution of [Utils].[ManageExtendedProperty] completed.',0,1);
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
    END;


END
GO


IF (@@ERROR = 0)
BEGIN
    PRINT '   PROCEDURE altered.';
END
ELSE
BEGIN
    PRINT '   Error while trying to alter procedure';
    RETURN
END;
GO

RAISERROR('-----------------------------------------------------------------------------------------------------------------',0,1);
RAISERROR('',0,1);
GO

