/*requires Schema.Docs.sql*/
/*requires Table.Docs.SchemaObjectDictionary.sql*/
/*requires Table.Docs.SchemaObjectsAttributes.sql*/

DECLARE @ProcedureSchema NVARCHAR(256);
DECLARE @ProcedureName   NVARCHAR(256);

SET @ProcedureSchema = 'Docs' ;
SET @ProcedureName = 'PopulateSchemaObjectDictionary' ;

PRINT '-----------------------------------------------------------------------------------------------------------------';
PRINT 'PROCEDURE [' + @ProcedureSchema + '].[' + @ProcedureName +  ']';

IF  NOT EXISTS (SELECT 1 FROM sys.objects WHERE object_id = OBJECT_ID(N'[' + @ProcedureSchema + N'].[' + @ProcedureName +  N']') AND type in (N'P'))
BEGIN
    BEGIN TRY
        EXECUTE ('CREATE Procedure [' + @ProcedureSchema + '].[' + @ProcedureName +  '] ( ' +
                ' @ServerName    varchar(512), ' +
                ' @DbName    varchar(50) ' +
                ') ' +
                'AS ' +
                'BEGIN ' +
                '   SELECT ''Not implemented'' ' +
                'END')
    END TRY
    BEGIN CATCH
        PRINT '   Error while trying to create procedure'
        RETURN
    END CATCH

    PRINT '   PROCEDURE created.'
END
GO

ALTER PROCEDURE [Docs].[PopulateSchemaObjectDictionary] (
    @ServerId                           BIGINT          = NULL,
    @DatabaseName                       VARCHAR(256)    = NULL,
    @SchemaName                         VARCHAR(256)    = NULL,
    --@OutputType                         VARCHAR(16)     = 'NONE', -- Possible values: NONE, REPORT, TABLE
    @ReadFromExtendedProperties         BIT             = 1,
    @ExtendedPropertyName               VARCHAR(256)    = 'MS_Description',
    @PreserveDescriptionInDictionary    BIT             = 1,
    @ReadonlyExecution                  BIT             = 0,
    @Debug                              BIT             = 0
)
AS
/*
  ===================================================================================
    DESCRIPTION:
        Populates/Updates SchemaObjectDictionary and SchemaObjectsAttributes tables
        data based on current 
        
        
    PARAMETERS:

    REQUIREMENTS:

    EXAMPLE USAGE :
    
        -- run with default (all databases, all schemas)
        EXEC [Docs].[PopulateSchemaObjectDictionary] 
                @Debug              = 1
        ;
        
        -- run and take only from a particular database
        EXEC [Docs].[PopulateSchemaObjectDictionary] 
                @DatabaseName       = 'CommonDatabase_DEV', 
                @Debug              = 1
        ;


        -- show results from INFORMATION_SCHEMA with extended properties MS_Description
        -- filter on database
		EXEC [Docs].[PopulateSchemaObjectDictionary] 
                @DatabaseName       = 'CommonDatabase_DEV', 
                @ReadonlyExecution  = 1, 
                @Debug              = 1
        ;
        
        -- show results from INFORMATION_SCHEMA with extended properties MS_Description
        -- filter on database AND schema
        EXEC [Docs].[PopulateSchemaObjectDictionary] 
                @DatabaseName       = 'CommonDatabase_DEV', 
                @SchemaName         = 'zzz', 
                @ReadonlyExecution  = 1, 
                @Debug              = 1
        ;
  ===================================================================================
*/
BEGIN
    SET NOCOUNT ON;
    DECLARE @tsql               nvarchar(max);
    DECLARE @LineFeed           CHAR(2);
    DECLARE @logMsg             VARCHAR(MAX);

    SELECT
        @tsql               = '',
        @LineFeed           = CHAR(13) + CHAR(10),
        @DatabaseName       = CASE WHEN LEN(LTRIM(RTRIM(@DatabaseName))) = 0 THEN NULL ELSE @DatabaseName  END,
        @SchemaName         = CASE WHEN LEN(LTRIM(RTRIM(@SchemaName))) = 0 THEN NULL ELSE @SchemaName  END
    ;

    if (@Debug = 1)
    BEGIN
        PRINT '-- -----------------------------------------------------------------------------------------------------------------';
        PRINT '-- Now running [Docs].[PopulateSchemaObjectDictionary] stored procedure.';
        PRINT '-- -----------------------------------------------------------------------------------------------------------------';
    END;

    IF(@DatabaseName IS NOT NULL AND DB_ID(@DatabaseName) IS NULL)
    BEGIN
        RAISERROR('Database [%s] not found',12,1,@DatabaseName);
        RETURN;
    END;
    
    IF(@ReadFromExtendedProperties = 1 AND (@ExtendedPropertyName IS NULL OR LEN(@ExtendedPropertyName) = 0))
    BEGIN
        RAISERROR('This procedure should read from extended properties but no property name provided',12,1);
        RETURN;
    END;
    
    if (@Debug = 1)
    BEGIN
        RAISERROR('-- Creating temporary objects',0,1);
    END;
    
    IF(OBJECT_ID('tempdb..#DiscoveredSchemaObjects') IS NOT NULL)
    BEGIN 
        EXEC sp_executesql N'DROP TABLE #DiscoveredSchemaObjects';
    END;
    
    IF(OBJECT_ID('tempdb..#DiscoveredObjectsAttributes') IS NOT NULL)
    BEGIN 
        EXEC sp_executesql N'DROP TABLE #DiscoveredObjectsAttributes;';
    END;
    
    CREATE TABLE #DiscoveredSchemaObjects (
        DatabaseName            VARCHAR(256)    NOT NULL,
        DbObjectId              BIGINT          NULL,
        SchemaName              VARCHAR(256)    NOT NULL,
        ObjectName              VARCHAR(256)    NOT NULL,
        ObjectType              VARCHAR(256)    NOT NULL,
        ObjectDescription       VARCHAR(4000)   NULL
    );
    
    CREATE TABLE #DiscoveredObjectsAttributes (
        DatabaseName            VARCHAR(256)    NOT NULL,
        DbObjectId              BIGINT          NOT NULL,
        AttributeName           VARCHAR(256)    NOT NULL,
        OrdinalPosition         INT             NOT NULL,
        PropertyDescription     VARCHAR(4000)   NULL
    );
    
    SET @tsql = 'WITH AllTables ' + @LineFeed +
                'AS (' + @LineFeed +
                '    SELECT  ' + @LineFeed +
                '        DB_NAME() as DbName,' + @LineFeed +
                '        OBJECT_ID(' + @LineFeed +
                '            QUOTENAME(TABLE_SCHEMA) + ''.'' + QUOTENAME(TABLE_NAME)' + @LineFeed +
                '        ) AS DbObjectId ,' + @LineFeed +
                '        TABLE_SCHEMA as SchemaName,' + @LineFeed +
                '        TABLE_NAME as ObjectName,' + @LineFeed +
                '        (SELECT type_desc FROM sys.objects WHERE object_id = OBJECT_ID(QUOTENAME(TABLE_SCHEMA) + ''.'' + QUOTENAME(TABLE_NAME))) as ObjectType,' + @LineFeed +
                '        NULL as ObjectDescription' + @LineFeed +
                '    FROM    ' + @LineFeed +
                '        INFORMATION_SCHEMA.TABLES t ' + @LineFeed +
                '    WHERE   ' + @LineFeed +
                '        table_Type = ''BASE TABLE''' + @LineFeed +
                CASE WHEN @SchemaName IS NULL THEN '' ELSE '    AND TABLE_SCHEMA = ''' + @SchemaName + '''' + @LineFeed END +
                '    AND OBJECTPROPERTY(' + @LineFeed +
                '            OBJECT_ID(' + @LineFeed +
                '                QUOTENAME(TABLE_SCHEMA) + ''.'' + QUOTENAME(TABLE_NAME)' + @LineFeed +
                '            ),' + @LineFeed +
                '            ''IsMSShipped''' + @LineFeed +
                '        ) = 0' + @LineFeed +
                '),' + @LineFeed +
                'AllViews ' + @LineFeed +
                'AS (' + @LineFeed +
                '    SELECT  ' + @LineFeed +
                '        DB_NAME() as DbName,' + @LineFeed +
                '        OBJECT_ID(' + @LineFeed +
                '            QUOTENAME(TABLE_SCHEMA) + ''.'' + QUOTENAME(TABLE_NAME)' + @LineFeed +
                '        ) AS DbObjectId ,' + @LineFeed +
                '        TABLE_SCHEMA as SchemaName,' + @LineFeed +
                '        TABLE_NAME as ObjectName,' + @LineFeed +
                '        ''VIEW'' as ObjectType,' + @LineFeed +
                '        NULL as ObjectDescription' + @LineFeed +
                '    FROM    ' + @LineFeed +
                '        INFORMATION_SCHEMA.VIEWS' + @LineFeed +
                '    WHERE   ' + @LineFeed +
                '        OBJECTPROPERTY(' + @LineFeed +
                '            OBJECT_ID(' + @LineFeed +
                '                QUOTENAME(TABLE_SCHEMA) + ''.'' + QUOTENAME(TABLE_NAME)' + @LineFeed +
                '            ),' + @LineFeed +
                '            ''IsMSShipped''' + @LineFeed +
                '        ) = 0' + @LineFeed +
                CASE WHEN @SchemaName IS NULL THEN '' ELSE '    AND TABLE_SCHEMA = ''' + @SchemaName + '''' + @LineFeed END +
                '),' + @LineFeed +
                'AllRoutines' + @LineFeed +
                'AS (' + @LineFeed +
                '    SELECT  ' + @LineFeed +
                '        DB_NAME() as DbName,' + @LineFeed +
                '        OBJECT_ID(' + @LineFeed +
                '            QUOTENAME(ROUTINE_SCHEMA) + ''.'' + QUOTENAME(ROUTINE_NAME)' + @LineFeed +
                '        ) AS DbObjectId ,' + @LineFeed +
                '        ROUTINE_SCHEMA as SchemaName,' + @LineFeed +
                '        ROUTINE_NAME as ObjectName,' + @LineFeed +
                '        ROUTINE_TYPE as ObjectType,' + @LineFeed +
                '        NULL as ObjectDescription' + @LineFeed +
                '    FROM    ' + @LineFeed +
                '        INFORMATION_SCHEMA.ROUTINES   ' + @LineFeed +
                '    WHERE   ' + @LineFeed +
                '        OBJECTPROPERTY(' + @LineFeed +
                '            OBJECT_ID(' + @LineFeed +
                '                QUOTENAME(ROUTINE_SCHEMA) + ''.'' + QUOTENAME(ROUTINE_NAME)' + @LineFeed +
                '            ),' + @LineFeed +
                '            ''IsMSShipped''' + @LineFeed +
                '        ) = 0' + @LineFeed +
                CASE WHEN @SchemaName IS NULL THEN '' ELSE '    AND ROUTINE_SCHEMA = ''' + @SchemaName + '''' + @LineFeed END +
                '),' + @LineFeed +
                'AllDomains' + @LineFeed +
                'AS (' + @LineFeed +
                '    SELECT' + @LineFeed +
                '        DB_NAME()          as DbName,' + @LineFeed +
                '        NULL               as ObjectId,' + @LineFeed +
                '        DOMAIN_SCHEMA      as SchemaName,' + @LineFeed +
                '        DOMAIN_NAME        as ObjectName,' + @LineFeed +
                '        UPPER(DATA_TYPE)   as ObjectType,' + @LineFeed +
                '        NULL               as ObjectDescription' + @LineFeed +
                '    FROM' + @LineFeed +
                '        INFORMATION_SCHEMA.DOMAINS' + @LineFeed +
                CASE WHEN @SchemaName IS NULL THEN '' ELSE '    WHERE DOMAIN_SCHEMA = ''' + @SchemaName + '''' + @LineFeed END +
                ')' + @LineFeed
                ;
                
    /* 
        TODO: Add Sequences for SQL Server 2012+ versions
    */
    
    SET @tsql = @tsql +
                'INSERT INTO #DiscoveredSchemaObjects' + @LineFeed +
                'select * FROM AllTables ' + @LineFeed +
                'UNION ALL' + @LineFeed +
                'SELECT * FROM AllViews' + @LineFeed +
                'UNION ALL' + @LineFeed +
                'SELECT * FROM AllRoutines' + @LineFeed +
                'UNION ALL' + @LineFeed +
                'SELECT * FROM AllDomains' + @LineFeed +
                /* TODO: Add AllSequences*/
                ';'
    ;
    
    if (@Debug = 1)
    BEGIN
        RAISERROR('-- Collecting all objects for further documentation',0,1);
    END;
    
    exec Common.RunQueryAcrossDatabases 
            @QueryTxt                   = @tsql , 
            @IncludeSystemDatabases     = 1, 
            @OnlyAccessibleDatabases    = 1, 
            @DbName_equals              = @DatabaseName,
            @Debug                      = @Debug
    ;
    
    if (@Debug = 1)
    BEGIN
        RAISERROR('-- Now collecting data object columns and programmable components parameters',0,1,@ExtendedPropertyName);
    END;
    
    SET @tsql = 'INSERT INTO #DiscoveredObjectsAttributes(' + @LineFeed +
                '    DatabaseName,DbObjectId,AttributeName,OrdinalPosition' + @LineFeed +
                ') ' + @LineFeed +
                'select ' + @LineFeed +
                '    DB_NAME() as DatabaseName,' + @LineFeed +
                '    objs.DbObjectId,' + @LineFeed +
                '    cols.COLUMN_NAME as AttributeName,' + @LineFeed +
                '    cols.ORDINAL_POSITION as OrdinalPosition' + @LineFeed +
                'From INFORMATION_SCHEMA.COLUMNS cols' + @LineFeed +
                'INNER JOIN #DiscoveredSchemaObjects objs' + @LineFeed +
                'ON cols.TABLE_CATALOG = objs.DatabaseName' + @LineFeed +
                'AND cols.TABLE_SCHEMA = objs.SchemaName' + @LineFeed +
                'AND cols.TABLE_NAME   = objs.ObjectName' + @LineFeed + 
                'UNION ALL' + @LineFeed +
                'SELECT' + @LineFeed +
                '    DB_NAME() as DatabaseName,' + @LineFeed +
                '    objs.DbObjectId,' + @LineFeed +
                '    parms.PARAMETER_NAME as AttributeName,' + @LineFeed +
                '    parms.ORDINAL_POSITION as OrdinalPosition' + @LineFeed +
                'FROM INFORMATION_SCHEMA.PARAMETERS parms' + @LineFeed +
                'INNER JOIN #DiscoveredSchemaObjects objs' + @LineFeed +
                'ON parms.SPECIFIC_CATALOG = objs.DatabaseName' + @LineFeed +
                'AND parms.SPECIFIC_SCHEMA = objs.SchemaName' + @LineFeed +
                'AND parms.SPECIFIC_NAME   = objs.ObjectName' + @LineFeed + 
                'WHERE parms.PARAMETER_NAME IS NOT NULL AND LEN(parms.PARAMETER_NAME) > 0' + @LineFeed 
                ;
    
    exec Common.RunQueryAcrossDatabases 
            @QueryTxt                   = @tsql , 
            @IncludeSystemDatabases     = 1, 
            @OnlyAccessibleDatabases    = 1, 
            @DbName_equals              = @DatabaseName,
            @Debug                      = @Debug
    ;
    
    
    IF(@ReadFromExtendedProperties = 1)
    BEGIN 
        if (@Debug = 1)
        BEGIN
            RAISERROR('-- Taking back existing documentation based on extended property with name [%s]',0,1,@ExtendedPropertyName);
        END;
        
        SET @tsql = 'update #DiscoveredSchemaObjects' + @LineFeed +
                    'set ObjectDescription = CONVERT(VARCHAR(MAX),p.value)' + @LineFeed +
                    'FROM ' + @LineFeed +
                    '    sys.extended_properties p' + @LineFeed +
                    'inner join ' + @LineFeed +
                    '    #DiscoveredSchemaObjects s' + @LineFeed +
                    'on  p.major_id    = s.DbObjectId' + @LineFeed +
                    'and p.minor_id    = 0 ' + @LineFeed +
                    'AND UPPER(p.name) = ''' + upper(@ExtendedPropertyName) + '''' + @LineFeed +
                    'WHERE s.DatabaseName = DB_NAME()' + @LineFeed
        ;
        
        exec Common.RunQueryAcrossDatabases 
                @QueryTxt                   = @tsql , 
                @IncludeSystemDatabases     = 1, 
                @OnlyAccessibleDatabases    = 1, 
                @DbName_equals              = @DatabaseName,
                @Debug                      = @Debug
        ;
        
        SET @tsql = 'update #DiscoveredObjectsAttributes' + @LineFeed +
                    'set PropertyDescription = CONVERT(VARCHAR(MAX),p.value)' + @LineFeed +
                    'FROM ' + @LineFeed +
                    '    sys.extended_properties p' + @LineFeed +
                    'inner join ' + @LineFeed +
                    '    #DiscoveredObjectsAttributes s' + @LineFeed +
                    'on  p.major_id    = s.DbObjectId' + @LineFeed +
                    'and p.minor_id    = s.OrdinalPosition ' + @LineFeed +
                    'AND UPPER(p.name) = ''' + upper(@ExtendedPropertyName) + '''' + @LineFeed +
                    'WHERE s.DatabaseName = DB_NAME()' + @LineFeed
        ;

        
        exec Common.RunQueryAcrossDatabases 
                @QueryTxt                   = @tsql , 
                @IncludeSystemDatabases     = 1, 
                @OnlyAccessibleDatabases    = 1, 
                @DbName_equals              = @DatabaseName,
                @Debug                      = @Debug
        ;        
        
    END;
    
    /*
    SELECT * FROM #DiscoveredSchemaObjects order by DatabaseName, DbObjectId;
    SELECT * FROM #DiscoveredObjectsAttributes where PropertyDescription IS NOT NULL order by DatabaseName, DbObjectId ;
    */
    
    
    IF(@Debug = 1)
    BEGIN
        SELECT @logMsg = 'There are ' + CONVERT(VARCHAR(10),COUNT_BIG(*)) + ' objects that have been found' + @LineFeed
        FROM #DiscoveredSchemaObjects ;
        
        SELECT @logMsg = @logMsg + '    accross ' + CONVERT(VARCHAR(10),COUNT_BIG(*)) + ' database(s)' + @LineFeed
        FROM (
            select distinct DatabaseName
            FROM #DiscoveredSchemaObjects
        ) a;
        
        RAISERROR(@logMsg,0,1);
    END;
    
    BEGIN TRANSACTION
    
    BEGIN TRY
        IF(@Debug = 1)
        BEGIN
            RAISERROR('Updating dictionary with collected data',0,1);
        END;
        
        IF(@ReadonlyExecution = 1)
        BEGIN
            SELECT 
                DatabaseName,
                SchemaName,
                ObjectName,
                ObjectType,
                ObjectDescription
            FROM #DiscoveredSchemaObjects ;
            
            SELECT dso.DatabaseName,dso.SchemaName,dso.ObjectName,dso.ObjectType,dso.DbObjectId,ddop.AttributeName,ddop.PropertyDescription as AttributeDescription
            FROM #DiscoveredSchemaObjects dso
            INNER JOIN #DiscoveredObjectsAttributes ddop
            ON dso.DbObjectId = ddop.DbObjectId 
        END;
        ELSE
        BEGIN
            MERGE Docs.SchemaObjectDictionary t
            USING (
                SELECT 
                    @ServerId as ServerId,
                    DatabaseName,
                    SchemaName,
                    ObjectName,
                    ObjectType,
                    ObjectDescription
                FROM #DiscoveredSchemaObjects
            ) i 
            ON ((t.ServerId IS NULL AND i.ServerId IS NULL) OR (t.ServerId = i.ServerId))
            AND (t.DatabaseName = i.DatabaseName)
            AND t.SchemaName = i.SchemaName
            and t.ObjectName = i.ObjectName
            WHEN MATCHED THEN 
                UPDATE 
                SET 
                    hasBeenDeleted          = 0,
                    DeletionDiscoveryDate   = NULL,
                    ObjectDescription       = CASE 
                                                WHEN t.ObjectDescription IS NULL AND i.ObjectDescription IS NOT NULL THEN i.ObjectDescription
                                                WHEN t.ObjectDescription IS NOT NULL AND i.ObjectDescription IS NULL THEN t.ObjectDescription
                                                ELSE CASE WHEN @PreserveDescriptionInDictionary = 1 THEN t.ObjectDescription
                                                          ELSE i.ObjectDescription
                                                     END
                                              END
            WHEN NOT MATCHED BY TARGET THEN
                INSERT (
                    ServerId,DatabaseName,SchemaName,ObjectName,ObjectType,ObjectDescription,isDepreciated,hasBeenDeleted
                )
                VALUES (
                    i.ServerId,i.DatabaseName,i.SchemaName,i.ObjectName,i.ObjectType,i.ObjectDescription,0,0
                )
            WHEN NOT MATCHED BY SOURCE THEN
                UPDATE
                set 
                    hasBeenDeleted          = 1, 
                    DeletionDiscoveryDate   = SYSDATETIME()
            ;
            
            WITH AllDataObjects 
            AS (
                SELECT 		DefinitionId, DatabaseName,SchemaName,ObjectName,ObjectType
                From Docs.SchemaObjectDictionary
                WHERE 
                    (ServerId = @ServerId OR (ServerId IS NULL AND @ServerId IS NULL))
                AND (HasBeenDeleted = 0   OR DeletionDiscoveryDate IS NOT NULL)
            )
            MERGE Docs.SchemaObjectsAttributes t
            USING (
                SELECT ado.DefinitionId, ado.DatabaseName,ado.SchemaName,ado.ObjectName,ado.ObjectType,dso.DbObjectId,ddop.AttributeName,ddop.PropertyDescription as AttributeDescription
                FROM AllDataObjects ado
                INNER JOIN #DiscoveredSchemaObjects dso
                ON ado.DatabaseName = dso.DatabaseName
                AND ado.SchemaName = dso.SchemaName
                AND ado.ObjectName = dso.ObjectName 
                INNER JOIN #DiscoveredObjectsAttributes ddop
                ON dso.DbObjectId = ddop.DbObjectId 
            ) i
            ON 
                t.ObjectDictionaryId = i.DefinitionId
            AND t.AttributeName = i.AttributeName
            WHEN MATCHED THEN 
                UPDATE 
                SET 
                    hasBeenDeleted          = 0,
                    DeletionDiscoveryDate   = NULL,
                    AttributeDescription    = CASE 
                                                WHEN t.AttributeDescription IS NULL AND i.AttributeDescription IS NOT NULL THEN i.AttributeDescription
                                                WHEN t.AttributeDescription IS NOT NULL AND i.AttributeDescription IS NULL THEN t.AttributeDescription
                                                ELSE CASE WHEN @PreserveDescriptionInDictionary = 1 THEN t.AttributeDescription
                                                          ELSE i.AttributeDescription
                                                     END
                                              END
            WHEN NOT MATCHED BY TARGET THEN
                INSERT (
                    ObjectDictionaryId,AttributeName,AttributeDescription
                )
                VALUES (
                    i.DefinitionId,i.AttributeName,i.AttributeDescription
                )
            WHEN NOT MATCHED BY  SOURCE THEN
                UPDATE 
                SET 
                    HasBeenDeleted = 1,
                    DeletionDiscoveryDate = SYSDATETIME()
            ;
                    
        END;
		COMMIT;
    END TRY
    BEGIN CATCH
        IF(@@TRANCOUNT > 0)
        BEGIN
            ROLLBACK;
        END;

        DECLARE @ErrorNumber    INT             = ERROR_NUMBER();
        DECLARE @ErrorLine      INT             = ERROR_LINE();
        DECLARE @ErrorMessage   NVARCHAR(4000)  = ERROR_MESSAGE();
        DECLARE @ErrorSeverity  INT             = ERROR_SEVERITY();
        DECLARE @ErrorState     INT             = ERROR_STATE();
        
     
        SET @logMsg = 'Actual error number: ' + CAST(@ErrorNumber AS VARCHAR(10));
        RAISERROR(@logMsg,0,1);
        SET @logMsg = 'Actual line number: ' + CAST(@ErrorLine AS VARCHAR(10));
        RAISERROR(@logMsg,0,1);
     
        RAISERROR(@ErrorMessage, @ErrorSeverity, @ErrorState);    
    END CATCH 
    
    
    
    if (@Debug = 1)
    BEGIN
        RAISERROR('Now performing cleanups',0,1);
    END;    
    
    IF(OBJECT_ID('tempdb..#DiscoveredSchemaObjects') IS NOT NULL)
    BEGIN 
        EXEC sp_executesql N'DROP TABLE #DiscoveredSchemaObjects';
    END;
    
    IF(OBJECT_ID('tempdb..#DiscoveredObjectsAttributes') IS NOT NULL)
    BEGIN 
        EXEC sp_executesql N'DROP TABLE #DiscoveredObjectsAttributes';
    END;

    if (@Debug = 1)
    BEGIN
        PRINT '-- -----------------------------------------------------------------------------------------------------------------';    
        PRINT '-- Execution of [Docs].[PopulateSchemaObjectDictionary] completed.';
        PRINT '-- -----------------------------------------------------------------------------------------------------------------';
    END;


END
GO


IF (@@ERROR = 0)
BEGIN
    PRINT '   PROCEDURE altered.';
END
ELSE
BEGIN
    PRINT '   Error while trying to alter procedure';
    RETURN
END;
GO

PRINT '-----------------------------------------------------------------------------------------------------------------';
PRINT '';
GO
