/*requires Schema.Reporting.sql*/

PRINT '--------------------------------------------------------------------------------------------------------------';
PRINT 'PROCEDURE [Reporting].[DeadlockTimeLineFromTable]';

IF  NOT EXISTS (SELECT 1 FROM sys.objects WHERE object_id = OBJECT_ID(N'[Reporting].[DeadlockTimeLineFromTable]') AND type in (N'P'))
BEGIN
    EXECUTE ('CREATE Procedure [Reporting].[DeadlockTimeLineFromTable] ( ' +
            ' @ServerName    varchar(512), ' +
            ' @DbName    varchar(50) ' +
            ') ' +
            'AS ' +
            'BEGIN ' +
            '   SELECT ''Not implemented'' ' +
            'END')
    ;
    
    IF @@ERROR = 0
        PRINT '   PROCEDURE created.';
    ELSE
    BEGIN
        PRINT '   Error while trying to create procedure';
        RETURN
    END  ;      
END;
GO

ALTER PROCEDURE [Reporting].[DeadlockTimeLineFromTable](
    @ReportTemplateName                     VARCHAR(256)    = NULL , -- "Hour" | "minute" | "Day" | NULL
    @SourceDatabaseName                     VARCHAR(256)    = NULL,
    @SourceSchemaName                       VARCHAR(256)    = NULL,
    @SourceTableName                        VARCHAR(256),
    @SourceTimeColumnName                   VARCHAR(256),
    @FilterStartDate                        DATETIME        = NULL,
    @FilterEndDate                          DATETIME        = NULL,
    @OutputType                             VARCHAR(32)     = 'NONE', -- NONE | TABLE | SCHEMA
    @OutputDatabaseName                     VARCHAR(256)    = NULL,
    @OutputSchemaName                       VARCHAR(256)    = NULL,
    @OutputTableName                        VARCHAR(256)    = NULL,
    @FillTimeGaps                           BIT             = 0,
    @Debug                                  BIT             = 1
)
AS
/*
    Example usages
    
    
        -- Display schema for output table
        EXEC [Reporting].[DeadlockTimeLineFromTable]
                @SourceSchemaName       = 'Reporting',
				@SourceTableName		= 'ShreddedDeadlocksHistory',
				@SourceTimeColumnName	= 'BatchCompleted',
				@ReportTemplateName		= 'Day',
				@OutputType				= 'SCHEMA',
				@Debug					= 1
        ;
    
        -- Day report from 20170420 with time gaps filling 
		-- No storage as a report
		EXEC [Reporting].[DeadlockTimeLineFromTable]
                @SourceSchemaName       = 'Reporting',
				@SourceTableName		= 'ShreddedDeadlocksHistory',
				@SourceTimeColumnName	= 'BatchCompleted',
				@ReportTemplateName		= 'Day',
				@FilterStartDate		= '2017-04-20',
				@OutputType				= 'TABLE',
				@FillTimeGaps			= 1,
				@Debug					= 1
        ;

		-- Hour report from 20170425 with time gaps filling 
		-- No storage as a report
		EXEC [Reporting].[DeadlockTimeLineFromTable]
                @SourceSchemaName       = 'Reporting',
				@SourceTableName		= 'ShreddedDeadlocksHistory',
				@SourceTimeColumnName	= 'BatchCompleted',
				@ReportTemplateName		= 'hour',
				@FilterStartDate		= '20170425',
				@OutputType				= 'TABLE',
				@FillTimeGaps			= 1,
				@Debug					= 1
        ;

		-- Minute report from 20170425 between 8 AM and 12 PM with time gaps filling 
		-- No storage as a report
		EXEC [Reporting].[DeadlockTimeLineFromTable]
                @SourceSchemaName       = 'Reporting',
				@SourceTableName		= 'ShreddedDeadlocksHistory',
				@SourceTimeColumnName	= 'BatchCompleted',
				@ReportTemplateName		= 'Minute',
				@FilterStartDate		= '2017-04-25 08:00:00.00',
				@FilterEndDate			= '2017-04-25 12:00:00.00',
				@OutputType				= 'TABLE',
				@FillTimeGaps			= 1,
				@Debug					= 1
        ;   
        
        -- Day report from 20170420 with time gaps filling 
		-- Store To Reporting.DeadlockTimeLine_Daily
        -- Still display results
		EXEC [Reporting].[DeadlockTimeLineFromTable]
                @SourceSchemaName       = 'Reporting',
				@SourceTableName		= 'ShreddedDeadlocksHistory',
				@SourceTimeColumnName	= 'BatchCompleted',
				@ReportTemplateName		= 'Day',
                @OutputSchemaName       = 'Reporting',
                @OutputTableName        = 'DeadlockTimeLine_Daily',
                @OutputType				= 'TABLE',
				@FilterStartDate		= '2017-04-20',
				@FillTimeGaps			= 1,
				@Debug					= 1
        ;

		-- Hour report from 20170425 with time gaps filling 
        -- Store To Reporting.DeadlockTimeLine_Hourly
        -- No results display
		EXEC [Reporting].[DeadlockTimeLineFromTable]
                @SourceSchemaName       = 'Reporting',
				@SourceTableName		= 'ShreddedDeadlocksHistory',
				@SourceTimeColumnName	= 'BatchCompleted',
				@ReportTemplateName		= 'hour',
                @OutputSchemaName       = 'Reporting',
                @OutputTableName        = 'DeadlockTimeLine_Hourly',
                @OutputType				= 'NONE',
				@FilterStartDate		= '20170425',
				@FillTimeGaps			= 1,
				@Debug					= 1
        ;

		-- Minute report from 20170425 between 8 AM and 12 PM with time gaps filling 
        -- Store To Reporting.DeadlockTimeLine_Hourly
        -- Still display results
		EXEC [Reporting].[DeadlockTimeLineFromTable]
                @SourceSchemaName       = 'Reporting',
				@SourceTableName		= 'ShreddedDeadlocksHistory',
				@SourceTimeColumnName	= 'BatchCompleted',
				@ReportTemplateName		= 'Minute',
				@FilterStartDate		= '2017-04-25 08:00:00.00',
				@FilterEndDate			= '2017-04-25 12:00:00.00',
                @OutputSchemaName       = 'Reporting',
                @OutputTableName        = 'DeadlockTimeLine_Minutely',
				@OutputType				= 'TABLE',
				@FillTimeGaps			= 1,
				@Debug					= 1
        ;
*/
BEGIN 
    SET NOCOUNT ON;
    
    DECLARE @cte_AllDeadlocks       NVARCHAR(MAX);
    DECLARE @cte_DeadlockByTime     NVARCHAR(MAX);
    DECLARE @cte_FillGaps           NVARCHAR(MAX);
    DECLARE @tsql                   NVARCHAR(MAX);
    DECLARE @tsql_OrderBy           NVARCHAR(MAX);
    DECLARE @LineFeed               CHAR(2);
    
    DECLARE @LogMsg                 VARCHAR(MAX);
    DECLARE @TmpCnt                 BIGINT;
    DECLARE @ExecRet                INT;
    
    DECLARE @ReportId               BIGINT;   
    DECLARE @ReportTitle            VARCHAR(1024);
    DECLARE @ReportParams           VARCHAR(MAX);
    
    DECLARE @OutputTableColumns     VARCHAR(MAX);
    DECLARE @SourceTableFullName    VARCHAR(1024);
    DECLARE @OutputTableFullName    VARCHAR(1024);
    DECLARE @NotTimeBound           BIT;
    
    SELECT
        @tsql                   = '',
        @tsql_OrderBy           = '',
        @cte_AllDeadlocks       = '',
        @cte_DeadlockByTime     = '',
        @cte_FillGaps           = '',
        @ReportParams           = '',
        @LineFeed               = CHAR(13) + CHAR(10),
        
        @OutputTableColumns     = '    [ReportId]            BIGINT           NOT NULL,' + @LineFeed +
                                  '    DateString            VARCHAR(32)      NOT NULL,' + @LineFeed +
                                  '    DeadlocksCount        BIGINT           NOT NULL,' + @LineFeed +
                                  '    RowOrder              BIGINT           NOT NULL' + @LineFeed ,
        -- Sanitize
        @OutputType             = UPPER(@OutputType),
        @ReportTemplateName     = LOWER(@ReportTemplateName),
        @OutputTableName        = CASE WHEN LEN(LTRIM(RTRIM(@OutputTableName))) = 0 THEN NULL ELSE @OutputTableName END,
        @SourceDatabaseName     = ISNULL(@SourceDatabaseName,DB_NAME()),
        @SourceSchemaName       = ISNULL(@SourceSchemaName,SCHEMA_NAME()),
        @OutputDatabaseName     = ISNULL(@OutputDatabaseName,DB_NAME()),
        @OutputSchemaName       = ISNULL(@OutputSchemaName,SCHEMA_NAME()),
        
        -- Deduct
        @ReportTitle            = 'Deadlock Timeline by ' + @ReportTemplateName,
        @SourceTableFullName    = QUOTENAME(@SourceDatabaseName) + '.' + QUOTENAME(@SourceSchemaName) + '.' + QUOTENAME(@SourceTableName),
        @OutputTableFullName    = QUOTENAME(@OutputDatabaseName) + '.' + QUOTENAME(@OutputSchemaName) + '.' + QUOTENAME(@OutputTableName),
        @NotTimeBound           = CASE WHEN @FilterStartDate IS NULL AND @FilterEndDate IS NULL THEN 1 ELSE 0 END
        
    ;
    
    if (@Debug = 1)
    BEGIN
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
        RAISERROR('-- Now running [Reporting].[TopDeadlocks] stored procedure.',0,1);
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
    END;
    
    IF(@OutputType = 'SCHEMA')
    BEGIN
        SELECT @OutputTableColumns ;
        RETURN;
    END;
    
    IF(@OutputType = 'NONE' AND (@OutputTableName IS NULL OR LEN(@OutputTableName) = 0))
    BEGIN
        RAISERROR('Either change value for @OutputType parameter or provide an output table name',12,1);
        RETURN;
    END;
    
    IF(@OutputType = 'TABLE' AND @OutputTableName IS NULL)
    BEGIN
        SET @FillTimeGaps = 1;
    END;
    
    IF(@SourceTimeColumnName IS NULL OR LEN(@SourceTimeColumnName) = 0)
    BEGIN
        RAISERROR('Please, provide a value for @SourceTimeColumnName',12,1) WITH NOWAIT;
        RETURN;
    END;
    
    IF( @ReportTemplateName NOT IN (
            'minute',
            'hour',
            'day'
        )
    )
    BEGIN
        RAISERROR('Unknown report template with name [%s]',12,1,@ReportTemplateName) WITH NOWAIT;
        RETURN;
    END;
    
    /*
        TODO: add a check for time... NO need to report on minutes for many days.. we won't be able to read graph
        
    IF(@ReportTemplateName = 'minute' 
        AND (
                @NotTimeBound = 1
            OR (@FilterStartDate IS NULL AND 
        )
        
    BEGIN
    
    END;
    */
    
    EXEC @ExecRet = Utils.CheckTableExists 
                            @DatabaseName       = @SourceDatabaseName,
                            @SchemaName         = @SourceSchemaName,
                            @TableName          = @SourceTableName,
                            @RunCreateTable     = 0,
                            @RunTruncateTable   = 0,
                            @Debug              = @Debug
    ;
    
    IF(@ExecRet <> 0)
    BEGIN
        RAISERROR('Source Table Name not found',12,1) WITH NOWAIT;
        RETURN;
    END;

    SET @ReportParams = @ReportParams + 
                        'Source Table IS "' + @SourceTableFullName + '"' + @LineFeed +
                        CASE 
                            WHEN @FilterStartDate IS NOT NULL AND @FilterEndDate IS NOT NULL
                                THEN 'Batch completion date BETWEEN "' + CONVERT(VARCHAR(32),@FilterStartDate,121) + '" AND "' + CONVERT(VARCHAR(32),@FilterEndDate,121) + '"' + @LineFeed 
                            WHEN @FilterStartDate IS NOT NULL AND @FilterEndDate IS NULL
                                THEN 'Batch completion date AFTER "' + CONVERT(VARCHAR(32),@FilterStartDate,121) + '"' + @LineFeed 
                            WHEN @FilterStartDate IS NULL AND @FilterEndDate IS NOT NULL
                                THEN 'Batch completion date BEFORE "' + CONVERT(VARCHAR(32),@FilterEndDate,121) + '"' + @LineFeed 
                            ELSE ''
                        END +
                        CASE 
                            WHEN @FillTimeGaps = 1 THEN 'Time Gaps are filled' + @LineFeed 
                            ELSE ''
                        END
                        ;
    
    IF(@Debug = 1)
    BEGIN
        RAISERROR('---------------------- ( Debug : Parameters summary ) -------------------------------',0,1);
        RAISERROR(@ReportParams ,0 ,1) WITH NOWAIT;
        RAISERROR('------------------------------- ( End Debug ) ---------------------------------------',0,1);
    END;
    
    /*
        TODO add a check for column @SourceTimeColumnName existence in source table
    */
    
    /*
        ==============================================================================================
        Building sub-queries
    */
    
    -- Building final sub-query
    IF(@OutputTableName IS NOT NULL)
    BEGIN 
        SET @tsql = @tsql + 
                    'INSERT INTO ' + @OutputTableFullName + @LineFeed
    END;
   
    
    SET @tsql = @tsql + 
                'SELECT' + @LineFeed
                ;
    
    IF(@OutputTableName IS NOT NULL)
    BEGIN
        SET @tsql = @tsql + 
                    '    @RptId,' + @LineFeed 
                    ;
    END;
    
    SET @tsql = @tsql + 
                '    r.DateStr,' + @LineFeed +
                '    ISNULL(r.DiscoveryCount,0) as DiscoveryCount,' + @LineFeed + 
                '    r.RowOrder' + @LineFeed +
                'FROM DeadlocksByTime as r' + @LineFeed
                ;
    
    /*
        at this point, we have everything except CTE to prepend in order to actually get deadlock
        This query in @tsql may be modified if @FillTimeGaps is set to 1:
            - we will need to RIGHT JOIN with the "calendar" CTE
    */
    
    SET @cte_AllDeadlocks =  @cte_AllDeadlocks +
                        'WITH AllDeadlocks' + @LineFeed +
                        'AS (' + @LineFeed +
                        '    SELECT' + @LineFeed +
                        '        DeadlockId,' + @LineFeed +
                        '        max('+ @SourceTimeColumnName +') as DeadlockTime,' + @LineFeed +
                        --'        DATEPART(YEAR,max('+ @SourceTimeColumnName +')) as LogYearNum,' + @LineFeed +
                        --'        DATEPART(MONTH,max('+ @SourceTimeColumnName +')) as LogMonthNum,' + @LineFeed +
                        --'        DATEPART(DAY,max('+ @SourceTimeColumnName +')) as LogDayNum,' + @LineFeed +
                        --'        DATEPART(MINUTE,max('+ @SourceTimeColumnName +')) as LogMinuteNum,' + @LineFeed +
                        '        convert(char(8),max('+ @SourceTimeColumnName +'), 112) as LogDayShort,' + @LineFeed +
                        '        REPLACE(STR(DATEPART(HOUR,max('+ @SourceTimeColumnName +')),2,0),'' '',''0'') as LogHour,' + @LineFeed +
                        '        REPLACE(STR(DATEPART(MINUTE,max('+ @SourceTimeColumnName +')),2,0),'' '',''0'') as LogMinute' + @LineFeed +
                        '    FROM ' + @SourceTableFullName + @LineFeed +
                        '    group by DeadlockId' + @LineFeed +
                        -- limit to time interval to limit the initial load
                        '    having max('+ @SourceTimeColumnName +') BETWEEN CONVERT(DATETIME,''' + CONVERT(VARCHAR(32),ISNULL(@FilterStartDate,'1900-01-01'),112) + ''',112) ' + @LineFeed +
                        '                                                AND CONVERT(DATETIME,''' + CONVERT(VARCHAR(32),ISNULL(@FilterEndDate,GETDATE()),121) + ''',121)' + @LineFeed +
                        ')' + @LineFeed
						;
                        
    
    SET @cte_DeadlockByTime =   ',DeadlocksByTime ' + @LineFeed  +
                                'AS (' + @LineFeed  +
                                '    select' + @LineFeed +
                                '        LogDayShort,' + @LineFeed 
                                ;
    
    IF(@FillTimeGaps = 1)
    BEGIN
        SET @cte_FillGaps = @cte_FillGaps +
                        ',Dates' + @LineFeed +
                        'AS (' + @LineFeed +
                        '    select' + @LineFeed +
                        '        CalendarDate,CONVERT(CHAR(4),CalendarYear)' + @LineFeed +
                        '        + RIGHT(''0000'' + CONVERT(VARCHAR(2),CalendarMonth),2)' + @LineFeed +
                        '        + RIGHT(''0000'' + CONVERT(VARCHAR(2),CalendarDay),2) as YearMonthDay' + @LineFeed +
                        '    From Common.TallyCalendar' + @LineFeed +
                        '    where CalendarDate BETWEEN CONVERT(DATETIME,''' + CONVERT(VARCHAR(32),ISNULL(@FilterStartDate,'1900-01-01'),112) + ''',112) ' + @LineFeed +
                        '                       AND CONVERT(DATETIME,''' + CONVERT(VARCHAR(32),ISNULL(@FilterEndDate,GETDATE()),121) + ''',121)' + @LineFeed +
                        ')' + @LineFeed 
                        ;
    END;      
    
    IF(@ReportTemplateName = 'minute')
    BEGIN
        SET @tsql_OrderBy = 'LogDayShort,LogHour,LogMinute';        
        SET @cte_DeadlockByTime =   @cte_DeadlockByTime +
                                        '        LogDayShort + '' @ '' + LogHour + '':'' + LogMinute as DateStr,' + @LineFeed +
                                        '        LogHour,' + @LineFeed  +
                                        '        LogMinute,' + @LineFeed  +
                                        '        count_big(*) as DiscoveryCount,' + @LineFeed  +
                                        '        ROW_NUMBER() OVER (order by ' + @tsql_OrderBy + ') as RowOrder' + @LineFeed +
                                        '    From AllDeadlocks' + @LineFeed  +
                                        '    group by ' + @tsql_OrderBy + @LineFeed  +
                                        ')' + @LineFeed  
                                        ;
        
        IF(@FillTimeGaps = 1)
        BEGIN
            SET @tsql_OrderBy = 'YearMonthDay,HourDigits,MinuteDigits';
            
            SET @cte_FillGaps = @cte_FillGaps +
                            ',Timing' + @LineFeed + 
                            'AS (' + @LineFeed +
                            '    SELECT'  + @LineFeed + 
                            '        TimeStdFormat,' + @LineFeed + 
                            '        RIGHT(''0000'' + CONVERT(VARCHAR(2),SUBSTRING(TimeDigitsOnly,1,2)),2)  as HourDigits, ' + @LineFeed +
                            '        RIGHT(''0000'' + CONVERT(VARCHAR(2),SUBSTRING(TimeDigitsOnly,3,2)),2)  as MinuteDigits' + @LineFeed +
                            '    FROM Common.TimeDimension' + @LineFeed +
                            '    where TimeStdFormat IS NOT NULL' + @LineFeed +
                            '    AND NumSeconds = 0' + @LineFeed +
                            ')' + @LineFeed +
                            ',DateTimes' + @LineFeed +
                            'AS (' + @LineFeed +
                            '    SELECT' + @LineFeed +
                            '        YearMonthDay + '' @ '' + HourDigits + '':'' + MinuteDigits as DateStr,' + @LineFeed +
                            '        MinuteDigits,' + @LineFeed +
                            '        HourDigits,' + @LineFeed +
                            '        YearMonthDay,' + @LineFeed +
                            '        ROW_NUMBER() OVER (order by ' + @tsql_OrderBy + ') as RowOrder' + @LineFeed +
                            '    FROM Dates , Timing' + @LineFeed +
                            '    WHERE DATEADD(MINUTE,CONVERT(INT,MinuteDigits),DATEADD(HOUR,CONVERT(INT,HourDigits),CalendarDate)) ' + @LineFeed +
                            '           <= CONVERT(DATETIME,''' + CONVERT(VARCHAR(32),ISNULL(@FilterEndDate,GETDATE()),121) + ''',121)' + @LineFeed +
                            ')' + @LineFeed
                            ;
        END;
    END;
    ELSE IF(@ReportTemplateName = 'hour')
    BEGIN
        SET @tsql_OrderBy = 'LogDayShort,LogHour';        
            
        SET @cte_DeadlockByTime =   @cte_DeadlockByTime +
                                    '        LogDayShort + '' @ '' + LogHour  as DateStr,' + @LineFeed +
                                    '        LogHour,' + @LineFeed  +
                                    '        count_big(*) as DiscoveryCount,' + @LineFeed  +
                                    '        ROW_NUMBER() OVER (order by ' + @tsql_OrderBy + ') as RowOrder' + @LineFeed +
                                    '    From AllDeadlocks' + @LineFeed  +
                                    '    group by ' + @tsql_OrderBy + @LineFeed  +
                                    ')' + @LineFeed  
                                    ;
        IF(@FillTimeGaps = 1)
        BEGIN
            SET @tsql_OrderBy = 'YearMonthDay,HourDigits';
            
            SET @cte_FillGaps = @cte_FillGaps +
                            ',Timing' + @LineFeed + 
                            'AS (' + @LineFeed +
                            '    SELECT TimeStdFormat,RIGHT(''0000'' + CONVERT(VARCHAR(2),SUBSTRING(TimeDigitsOnly,1,2)),2)  as HourDigits' + @LineFeed +
                            '    FROM Common.TimeDimension' + @LineFeed +
                            '    where TimeStdFormat IS NOT NULL' + @LineFeed +
                            '    AND NumMinutes = 0' + @LineFeed +
                            '    AND NumSeconds = 0' + @LineFeed +
                            ')' + @LineFeed +
                            ',DateTimes' + @LineFeed +
                            'AS (' + @LineFeed +
                            '    SELECT' + @LineFeed +
                            '        YearMonthDay + '' @ '' + HourDigits as DateStr,' + @LineFeed +
                            '        HourDigits,' + @LineFeed +
                            '        YearMonthDay,' + @LineFeed +
                            '        ROW_NUMBER() OVER (order by ' + @tsql_OrderBy + ') as RowOrder' + @LineFeed +
                            '    FROM Dates , Timing' + @LineFeed +
                            '    WHERE DATEADD(HOUR,CONVERT(INT,HourDigits),CalendarDate) ' + @LineFeed +
                            '           <= CONVERT(DATETIME,''' + CONVERT(VARCHAR(32),ISNULL(@FilterEndDate,GETDATE()),121) + ''',121)' + @LineFeed +
                            ')' + @LineFeed
                            ;
        END;
    END
    ELSE IF (@ReportTemplateName = 'day')
    BEGIN
        SET @tsql_OrderBy = 'LogDayShort';        
    
        SET @cte_DeadlockByTime =   @cte_DeadlockByTime +
                                    '        LogDayShort  as DateStr,' + @LineFeed +
                                    '        count_big(*) as DiscoveryCount,' + @LineFeed  +
                                    '        ROW_NUMBER() OVER (order by ' + @tsql_OrderBy + ') as RowOrder' + @LineFeed +
                                    '    From AllDeadlocks' + @LineFeed  +
                                    '    group by ' + @tsql_OrderBy + @LineFeed  +
                                    ')' + @LineFeed  
                                    ;        
                                    
        IF(@FillTimeGaps = 1)
        BEGIN
            SET @tsql_OrderBy = 'YearMonthDay' ;
            
            SET @cte_FillGaps = @cte_FillGaps +
                            ',Timing' + @LineFeed + 
                            'AS (' + @LineFeed +
                            '    SELECT 0 as DayDigits' + @LineFeed +
                            ')' + @LineFeed +
                            ',DateTimes' + @LineFeed +
                            'AS (' + @LineFeed +
                            '    SELECT' + @LineFeed +
                            '        YearMonthDay as DateStr,' + @LineFeed +
                            '        YearMonthDay,' + @LineFeed +
                            '        ROW_NUMBER() OVER (order by ' + @tsql_OrderBy + ') as RowOrder' + @LineFeed +
                            '    FROM Dates , Timing' + @LineFeed +
                            ')' + @LineFeed
                            ;
        END;
    END;
    
    IF(@FillTimeGaps = 1)
    BEGIN
    
        /*
            We need to add the right join we mention above HERE.
            Why here and not in each condition?
                Because we used the same names for all CTEs and for joining columns
        */
        -- we also need to change the selected value for "DateStr" otherwize, we would get NULL for all gaps
        -- this has to be done before adding the right join
        SET @tsql = REPLACE(@tsql,'r.DateStr','dt.DateStr');
        SET @tsql = REPLACE(@tsql,'r.RowOrder','dt.RowOrder');
        
        SET @tsql = @tsql + 
                    'RIGHT JOIN DateTimes as dt' + @LineFeed + 
                    'ON r.DateStr = dt.DateStr' + @LineFeed +
                    CASE WHEN LEN(@tsql_OrderBy) = 0 THEN '' ELSE 'order by ' + @tsql_OrderBy + @LineFeed END 
                    ; 
    END;
    
    -- Complete @tsql with all @cte_* + @tsql
    
    IF(@Debug = 1)
    BEGIN
        RAISERROR('-- -------------------------------------------- ( Query to get deadlocks ) --------------------------------------',0,1) WITH NOWAIT ;
        RAISERROR(@cte_AllDeadlocks,0,1) WITH NOWAIT ;
        RAISERROR(@cte_DeadlockByTime,0,1) WITH NOWAIT ;
        RAISERROR(@cte_FillGaps,0,1) WITH NOWAIT ;
        RAISERROR(@tsql,0,1) WITH NOWAIT ;
        RAISERROR('-- ----------------------------------------- ( End of Query to get deadlocks ) ----------------------------------',0,1) WITH NOWAIT ;
    END;
    
    SET @tsql = @cte_AllDeadlocks + 
                @cte_DeadlockByTime +
                @cte_FillGaps +
                @tsql
                ;
                
    /*
    Now, we have the query to get back the results, let's just display it for Debug
    */
    
    /*
        ==============================================================================================
        Starting execution
    */
    IF(@Debug = 1)
    BEGIN
        RAISERROR('Starting actual execution',0,1) WITH NOWAIT;
    END;
    
    BEGIN TRY 
        IF(@OutputTableName IS NOT NULL AND LEN(@OutputTableName) > 0)
        BEGIN
        
            IF(@Debug = 1)
            BEGIN
                RAISERROR('Logging process execution to Reporting.ExecutionLog table',0,1) WITH NOWAIT;
            END;
        
            EXEC @ExecRet = [Reporting].[LogNewExecution] 
                @ReportName  = @ReportTitle,
                @ReportClass = 'Deadlock',
                @ReportId	 = @ReportId OUTPUT,
                @OutputType  = 'TABLE',
                @OutputPath  = @OutputTableFullName,
                @Debug       = @Debug
            ;
            
            IF(@ReportId IS NULL OR @ExecRet <> 0)
            BEGIN
                RAISERROR('Unable to log execution to Reporting.ExecutionLog table',12,1) WITH NOWAIT;
                RETURN;
            END;
            
            -- Update contents (ReportParams)
            UPDATE Reporting.ExecutionLog
            SET 
                ReportParams    = @ReportParams
            WHERE 
                ReportId = @ReportId
            ;
        END;
    
        IF(@ReportId IS NOT NULL)
        BEGIN
            -- check and create output table
            EXEC @ExecRet = Utils.CheckTableExists 
                                    @DatabaseName       = @OutputDatabaseName,
                                    @SchemaName         = @OutputSchemaName,
                                    @TableName          = @OutputTableName,
                                    @RunCreateTable     = 1,
                                    @TableColumnsDef    = @OutputTableColumns,
                                    @RunTruncateTable   = 0,
                                    @Debug              = @Debug
                            ;               
                            
            IF(@ExecRet <> 0)
            BEGIN
                RAISERROR('Output Table Name not found or could not be created',12,1) WITH NOWAIT;
                RETURN;
            END;
        END;
    
        -- run query
        EXEC @ExecRet = sp_executesql @tsql, N'@RptId BIGINT', @RptId = @ReportId ;
        
        IF(@ExecRet <> 0)
        BEGIN 
            RAISERROR('An error occurred during query execution',12,1) WITH NOWAIT;
        END;
        
        IF(@ReportId IS NOT NULL)
        BEGIN
            UPDATE Reporting.ExecutionLog
            SET 
                EndTime         = GETDATE(),
                Outcome         = 'SUCCESS'
            WHERE 
                ReportId = @ReportId
            ;
        END;
        
        IF(@OutputType = 'TABLE' AND @ReportId IS NOT NULL)
        BEGIN
            SET @tsql = 'SELECT * FROM ' + @OutputTableFullName + ' WHERE ReportId = @RptId ORDER BY RowOrder ASC';
            exec sp_executesql @tsql , N'@RptId BIGINT', @RptId = @ReportId ;
        END; 
        
    END TRY 
    BEGIN CATCH
        SELECT @LogMsg = ' ----------------------------- ( Details of the error caught during execution ) -----------------------------' + @LineFeed +
                         'Error #' + CONVERT(VARCHAR(10),ERROR_NUMBER()) + ' with Severity ' + CONVERT(VARCHAR(10),ERROR_SEVERITY()) + ', State ' + CONVERT(VARCHAR(10),ERROR_STATE()) + @LineFeed +
                         'in stored procedure ' + ISNULL(ERROR_PROCEDURE(),ISNULL(OBJECT_SCHEMA_NAME(@@PROCID) + '.' + OBJECT_NAME(@@PROCID),'N/A')) + @LineFeed +
                         'Message:' + @LineFeed +
                         ISNULL(ERROR_MESSAGE(),'N/A') + @LineFeed +
                         ' -------------------------- ( End of Details of the error caught during execution ) --------------------------' + @LineFeed
                        ;
        RAISERROR(@LogMsg,10,1);
    
        IF(@ReportId IS NOT NULL)
        BEGIN
            UPDATE Reporting.ExecutionLog
            SET 
                EndTime         = GETDATE(),
                Outcome         = 'FAILED', 
                ErrorDetails    = @LogMsg
            WHERE 
                ReportId = @ReportId
            ;
            
            SET @tsql = 'DELETE FROM ' +  @OutputTableFullName + ' WHERE ReportId = @RptId';
            EXEC sp_executesql @tsql ;
            
        END;
    END CATCH

END;
GO

IF @@ERROR = 0
    PRINT '   PROCEDURE altered.';
ELSE
BEGIN
    PRINT '   Error while trying to alter procedure';
    RETURN;
END ;
GO 

PRINT '--------------------------------------------------------------------------------------------------------------';
PRINT '' ;
GO