/*requires Schema.Reporting.sql*/

PRINT '--------------------------------------------------------------------------------------------------------------';
PRINT 'PROCEDURE [Reporting].[DeadlockTimeLine]';

IF  NOT EXISTS (SELECT 1 FROM sys.objects WHERE object_id = OBJECT_ID(N'[Reporting].[DeadlockTimeLine]') AND type in (N'P'))
BEGIN
    EXECUTE ('CREATE Procedure [Reporting].[DeadlockTimeLine] ( ' +
            ' @ServerName    varchar(512), ' +
            ' @DbName    varchar(50) ' +
            ') ' +
            'AS ' +
            'BEGIN ' +
            '   SELECT ''Not implemented'' ' +
            'END')
	;
    
	IF @@ERROR = 0
        PRINT '   PROCEDURE created.';
    ELSE
    BEGIN
        PRINT '   Error while trying to create procedure';
        RETURN
    END  ;      
END;
GO

ALTER PROCEDURE [Reporting].[DeadlockTimeLine](
    @OnlyCurrentLog		                    BIT         = 0,
    @ErrorLogID			                    INT         = NULL,
    @StartDatePivot		                    DATETIME   = NULL,
    @EndDatePivot		                    DATETIME   = NULL,
    @TargetDatabaseName4OccurrencesReport   SYSNAME     = NULL,
    @TargetSchemaName4OccurrencesReport     SYSNAME     = NULL,
    @TargetTableName4OccurrencesReport      SYSNAME     = NULL,
    @AppendMode4OccurrencesReport           BIT         = 0,
    @TargetDatabaseName4HourlyReport        SYSNAME     = NULL,
    @TargetSchemaName4HourlyReport          SYSNAME     = NULL,
    @TargetTableName4HourlyReport           SYSNAME     = NULL,
    @AppendMode4HourlyReport                BIT         = 0,    
    @TargetDatabaseName4DailyReport         SYSNAME     = NULL,
    @TargetSchemaName4DailyReport           SYSNAME     = NULL,
    @TargetTableName4DailyReport            SYSNAME     = NULL,
    @AppendMode4DailyReport                 BIT         = 0,
    @Debug                                  BIT         = 1
)
AS
/*
    Example usages
    
        -- Analyze every event logs and display results
        EXEC [Reporting].[DeadlockTimeLine]
        
        -- Analyze current event log ONLY
        EXEC [Reporting].[DeadlockTimeLine] 
                    @OnlyCurrentLog = 1;

        -- Analyze reports from a given date
        EXEC [Reporting].[DeadlockTimeLine] 
                    @StartDatePivot  = '2016-01-01 010:00:00.000' ;

        -- Analyze reports in a given date range
        EXEC [Reporting].[DeadlockTimeLine] 
                    @StartDatePivot  = '2016-01-01 010:00:00.000' ,
                    @EndDatePivot  = '2016-01-04 010:00:00.000' ;

        -- Analyze reports until a given date
        EXEC [Reporting].[DeadlockTimeLine] 
                    @EndDatePivot  = '2016-01-04 010:00:00.000' ;

        -- Get report for every log file and store to tables
        -- Note: This will truncate the tables if they exist
        --       You can avoid using Database and Schema specs.. It will use the current Database and user default schema in that database.
        EXEC [Reporting].[DeadlockTimeLine] 
                    @TargetDatabaseName4HourlyReport = 'SAIDBA',
                    @TargetSchemaName4HourlyReport   = 'dbo',
                    @TargetTableName4HourlyReport    = 'DeadlockHourlyReport',
                    @TargetDatabaseName4DailyReport  = 'SAIDBA',
                    @TargetSchemaName4DailyReport    = 'dbo',
                    @TargetTableName4DailyReport     = 'DeadlockDailyReport'
        ;
        
        -- Analyze:
        select * from dbo.DeadlockDailyReport order by LogDateShort asc;
		select * from dbo.DeadlockHourlyReport order by 1 asc;
		select DATENAME(dw,CONVERT(DATETIME,LogDateShort,112)), avg(DeadlockCount) 
		from dbo.DeadlockDailyReport
		group by DATENAME(dw,CONVERT(DATETIME,LogDateShort,112))
		;
        
        select SUBSTRING(LogDateWithHour,9,2), avg(DeadlockCount) 
		from dbo.DeadlockHourlyReport
		group by SUBSTRING(LogDateWithHour,9,2)
		order by 1  
        
        select 
			DATEPART(dw,CONVERT(DATETIME,SUBSTRING(LogDateWithHour,0,9),112)) as DayID, 
			DATENAME(dw,CONVERT(DATETIME,SUBSTRING(LogDateWithHour,0,9),112)) as DayOfWeek,
			SUBSTRING(LogDateWithHour,9,2)									  as Hrs, 
			avg(DeadlockCount)												  as AverageDeadlockCount
		from dbo.DeadlockHourlyReport
		group by 
			DATEPART(dw,CONVERT(DATETIME,SUBSTRING(LogDateWithHour,0,9),112)),
			DATENAME(dw,CONVERT(DATETIME,SUBSTRING(LogDateWithHour,0,9),112)),
			SUBSTRING(LogDateWithHour,9,2)
		order by 1  ;
*/
BEGIN 
    SET NOCOUNT ON;
    
    DECLARE @tsql				NVARCHAR(MAX);
    DECLARE @TmpCnt				INT;
    DECLARE @logMsg             VARCHAR(MAX);
    DECLARE @LineFeed			CHAR(2);
    
    SET @LineFeed = CHAR(13) + CHAR(10);
    
    IF(OBJECT_ID('tempdb..#AllErrorLogs') IS NOT NULL)
    BEGIN
        exec sp_executesql N'DROP TABLE #AllErrorLogs';
    END;

    IF(OBJECT_ID('tempdb..#ErrorLogsContent') IS NOT NULL)
    BEGIN
        exec sp_executesql N'DROP TABLE #ErrorLogsContent';
    END;

    if(@Debug = 1)
    BEGIN
        RAISERROR('Existing temporary tables used in this procedure may have been dropped.',0,1);
    END;

    CREATE TABLE #AllErrorLogs (
        ArchiveID		INT NOT NULL
            CONSTRAINT PK_AllErrorLogs
                PRIMARY KEY CLUSTERED (ArchiveID),
        LastLogDate	    DATETIME,
        LogFileSizeKb	BIGINT
    );

    CREATE TABLE #ErrorLogsContent (
        EntryID		INT IDENTITY(1,1) NOT NULL
                            CONSTRAINT PK_#ErrorLogsContent
                                PRIMARY KEY CLUSTERED (EntryID),
        LogDate		DATETIME, 
        ProcessInfo NVARCHAR(100), 
        [LogText]	NVARCHAR(MAX),
        LogYear    as DATEPART(YEAR,LogDate),
        LogMonth   as DATEPART(MONTH,LogDate),
        LogDayNum  as DATEPART(DAY,LogDate),
        LogDayShort as convert(char(8), LogDate, 112) PERSISTED,
        LogHour    as REPLACE(STR(DATEPART(HOUR,LogDate),2,0),' ','0') PERSISTED,
        LogMinute  as DATEPART(MINUTE,LogDate)
    );

    if(@Debug = 1)
    BEGIN
        RAISERROR('Temporary tables created.',0,1);
    END;
    
    BEGIN TRY
        if(@Debug = 1)
        BEGIN
            RAISERROR('Now getting back error logs list.',0,1);
        END;

        INSERT INTO #AllErrorLogs	
            exec xp_enumerrorlogs
        ;

        if(@Debug = 1)
        BEGIN
            SELECT @TmpCnt = COUNT(*) from #AllErrorLogs;

            SET @logMsg = 'Total number of error logs: ' + CONVERT(VARCHAR(32),@TmpCnt);
            RAISERROR(@logMsg,0,1);
            RAISERROR('Now deleting unnecessary logs.',0,1);
        END;


        IF(@OnlyCurrentLog = 1)
        BEGIN
            if(@Debug = 1)
            BEGIN
                RAISERROR('Only current error log with be kept.',0,1);
            END;

            DELETE FROM #AllErrorLogs WHERE ArchiveID <> 0;
        END;
        ELSE IF(@ErrorLogID IS NOT NULL)
        BEGIN
            if(@Debug = 1)
            BEGIN
                SET @logMsg = 'OnlySpecified error log ID (' + CONVERT(VARCHAR(32),@ErrorLogID) + ') will be kept.';
                RAISERROR(@logMsg,0,1);
            END;
            
            DELETE FROM #AllErrorLogs 
            WHERE ArchiveID <> @ErrorLogID
            ;
        END;

        -- Filter on dates
        -- Get 
        WITH LogDates(ArchiveID,FirstLogDate)
        AS (
            select l2.ArchiveID , l1.LastLogDate
            From 
                #AllErrorLogs l1 
            inner join 
                #AllErrorLogs l2 
            on l1.ArchiveID = l2.ArchiveID + 1
        ) 
        DELETE FROM #AllErrorLogs
        WHERE ArchiveID NOT IN (
            SELECT 
                el.ArchiveID
            FROM #AllErrorLogs el
            LEFT JOIN
                LogDates ld
            ON el.ArchiveID = ld.ArchiveID
            WHERE 
                1 = CASE
                    WHEN (@StartDatePivot IS NULL AND @EndDatePivot IS NULL) 
                        -- everything
                    THEN 1
                    WHEN (	@StartDatePivot IS NULL 
                        AND @EndDatePivot IS NOT NULL 
                    ) 
                        -- everything until
                    THEN 
                        CASE WHEN(
                                @EndDatePivot BETWEEN ISNULL(ld.FirstLogDate,'1/1/1900 12:00:00') AND el.LastLogDate
                            OR	@EndDatePivot >= el.LastLogDate
                        )
                            THEN 1
                        ELSE 0
                        END
                    WHEN (	@StartDatePivot IS NOT NULL 
                        AND @EndDatePivot IS NULL 
                    )
                        -- everything from
                    THEN 
                        CASE WHEN(
                                @StartDatePivot BETWEEN ISNULL(ld.FirstLogDate,'1/1/1900 12:00:00') AND el.LastLogDate
                            OR	@StartDatePivot <= ld.FirstLogDate
                        )
                            THEN 1
                        ELSE 0
                        END
                    ELSE
                        -- interval provided
                        CASE WHEN(
                            (
                                @StartDatePivot BETWEEN ISNULL(ld.FirstLogDate,'1/1/1900 12:00:00') AND el.LastLogDate
                            OR	@StartDatePivot <= ld.FirstLogDate
                            )
                            AND (
                                @EndDatePivot BETWEEN ISNULL(ld.FirstLogDate,'1/1/1900 12:00:00') AND el.LastLogDate
                            OR	@EndDatePivot >= el.LastLogDate
                            )
                        )
                            THEN 1
                        ELSE 0
                        END
                END
        )
        ;

        if(@Debug = 1)
        BEGIN
            SELECT @TmpCnt = COUNT(*) from #AllErrorLogs;
            
            SET @logMsg = 'Total number of error logs after filtering: ' + CONVERT(VARCHAR(32),@TmpCnt);
            RAISERROR(@logMsg,0,1);
        END;


        -- Loop on Logs
        DECLARE @CurrentLogID INT;
        SELECT @CurrentLogID = min(ArchiveID)
        FROM #AllErrorLogs;

        WHILE (@CurrentLogID iS NOT NULL)
        BEGIN
            
            if(@Debug = 1)
            BEGIN
                SET @logMsg = 'Now loading log with ID: ' + CONVERT(VARCHAR(32),@CurrentLogID);
                RAISERROR(@logMsg,0,1);
            END;

            INSERT INTO #ErrorLogsContent
            exec xp_readerrorlog @CurrentLogID, 1;

            SELECT @CurrentLogID = min(ArchiveID)
            FROM #AllErrorLogs
            WHERE ArchiveID > @CurrentLogID;
        END;

        --SELECT * from #ErrorLogsContent ;

        if(@Debug = 1)
        BEGIN
            SELECT @TmpCnt = COUNT(*) from #ErrorLogsContent;
            
            SET @logMsg = 'Total number of rows extracted from error logs of interest: ' + CONVERT(VARCHAR(32),@TmpCnt);
            RAISERROR(@logMsg,0,1);
        END;

        -- Filtering

        if(@StartDatePivot is not null)
        BEGIN
            DELETE FROM #ErrorLogsContent
            WHERE LogDate < @StartDatePivot ;
        END;

        if(@EndDatePivot is not null)
        BEGIN
            DELETE FROM #ErrorLogsContent
            WHERE LogDate > @EndDatePivot ;
        END;

        DELETE FROM #ErrorLogsContent
        WHERE LogText not like 'Transaction (Process ID %) was deadlocked on % resources with another process and has been chosen as the deadlock victim. Rerun the transaction.'
        ;

        if(@Debug = 1)
        BEGIN
            SELECT @TmpCnt = COUNT(*) from #ErrorLogsContent;
            
            SET @logMsg = 'Total number of rows of interest from error logs of interest: ' + CONVERT(VARCHAR(32),@TmpCnt);
            RAISERROR(@logMsg,0,1);
        END;        
        
        --SELECT * from #ErrorLogsContent ;
        
        if(@Debug = 1)
        BEGIN
            RAISERROR('Now trying to complete Storage of the extracted data to tables (if necessary)',0,1);
        END;
        
        /*
        -- TODO : check to see if possible to run it inside stored procedure (for code reuse)
        -- Storage - Target Table - Hourly Report - PRE-Checks
        */

        
        -- ========== Occurrence Report 
                
        if(@Debug = 1)
        BEGIN
            RAISERROR('Performing some checks before taking care of occurrences report',0,1);
        END;
        
        if(@TargetDatabaseName4OccurrencesReport IS NULL)
        BEGIN
            SELECT @TargetDatabaseName4OccurrencesReport = DB_NAME();
        END;
        
        if(DB_ID(@TargetDatabaseName4OccurrencesReport) IS NULL) -- no QUOTENAME (it's quoted inside DB_ID)
        BEGIN
            RAISERROR('Database %s does not exist on server.',12,1,@TargetDatabaseName4OccurrencesReport);
        END;
        
        if(@TargetSchemaName4OccurrencesReport is null)
        BEGIN
            SELECT @TargetSchemaName4OccurrencesReport =SCHEMA_NAME();
        END;
        
        -- check schema exists

        SET @tsql = 'SELECT @cnt = count(*) from ' + QUOTENAME(@TargetDatabaseName4OccurrencesReport) + '.sys.schemas where name = @Name';
        exec sp_executesql @tsql , N'@cnt INT OUTPUT, @Name VARCHAR(256)', @cnt = @TmpCnt OUTPUT, @Name = @TargetSchemaName4OccurrencesReport;
        if(@TmpCnt = 0)
        BEGIN
            RAISERROR('Schema %s does not exist in %s database',12,1,@TargetSchemaName4OccurrencesReport,@TargetDatabaseName4OccurrencesReport);
        END;

        -- Now we know that the database and schema exists.
        -- if the tablename is set ==> need to store data
        -- if not ... forgive it !
        
        if(@TargetTableName4OccurrencesReport is not null)
        BEGIN
            -- create table if not exist.
            SET @TmpCnt = null;

            SET @tsql = 'Use ' + QUOTENAME(@TargetDatabaseName4OccurrencesReport) + '; SELECT @ObjID = OBJECT_ID(''' + QUOTENAME(@TargetSchemaName4OccurrencesReport) + '.' + QUOTENAME(@TargetTableName4OccurrencesReport) + ''')';

            exec sp_executesql @tsql , N'@ObjID INT OUTPUT', @ObjID = @TmpCnt OUTPUT;

            if(@TmpCnt is null)
            BEGIN
                SET @tsql = 'CREATE TABLE ' + QUOTENAME(@TargetDatabaseName4OccurrencesReport) + '.' + QUOTENAME(@TargetSchemaName4OccurrencesReport) + '.' + QUOTENAME(@TargetTableName4OccurrencesReport) + '(' +@LineFeed +
                            '    LogDateStamp     VARCHAR(32),' + @LineFeed +
                            '    DeadlockCount    INT' + @LineFeed +
                            ')';
                exec sp_executesql @tsql;
                if(@Debug = 1)
                BEGIN
                    SET @logMsg = 'Table ' + QUOTENAME(@TargetDatabaseName4OccurrencesReport) + '.' + QUOTENAME(@TargetSchemaName4OccurrencesReport) + '.' + QUOTENAME(@TargetTableName4OccurrencesReport) + ' created.';
                    RAISERROR(@logMsg,0,1);
                END;
            END;

            -- now the table is created.
            
            IF(@AppendMode4OccurrencesReport = 0)
            BEGIN
                SET @tsql = 'TRUNCATE TABLE ' + QUOTENAME(@TargetDatabaseName4OccurrencesReport) + '.' + QUOTENAME(@TargetSchemaName4OccurrencesReport) + '.' + QUOTENAME(@TargetTableName4OccurrencesReport);
                exec sp_executesql @tsql;
                
                if(@Debug = 1)
                BEGIN
                    SET @logMsg = 'Table ' + QUOTENAME(@TargetDatabaseName4OccurrencesReport) + '.' + QUOTENAME(@TargetSchemaName4OccurrencesReport) + '.' + QUOTENAME(@TargetTableName4OccurrencesReport) + ' truncated.';
                    RAISERROR(@logMsg,0,1);
                END;
            END;
            
            SET @tsql = 'INSERT INTO ' + QUOTENAME(@TargetDatabaseName4OccurrencesReport) + '.' + QUOTENAME(@TargetSchemaName4OccurrencesReport) + '.' + QUOTENAME(@TargetTableName4OccurrencesReport) + @LineFeed +
                        'SELECT CONVERT(VARCHAR(32),LogDate,120),count(*) FROM #ErrorLogsContent group by CONVERT(VARCHAR(32),LogDate,120)';

            exec sp_executesql @tsql;
        END;
        
        -- ========== Hourly Report 
        
        if(@Debug = 1)
        BEGIN
            RAISERROR('Performing some checks before taking care of Hourly report',0,1);
        END;

        if(@TargetDatabaseName4HourlyReport is null)
        BEGIN
            SELECT @TargetDatabaseName4HourlyReport = DB_NAME();
        END;

        if(DB_ID(@TargetDatabaseName4HourlyReport) IS NULL) -- no QUOTENAME (it's quoted inside DB_ID)
        BEGIN
            RAISERROR('Database %s does not exist on server.',12,1,@TargetDatabaseName4HourlyReport);
        END;

        if(@TargetSchemaName4HourlyReport is null)
        BEGIN
            SELECT @TargetSchemaName4HourlyReport =SCHEMA_NAME();
        END;

        -- check schema exists

        SET @tsql = 'SELECT @cnt = count(*) from ' + QUOTENAME(@TargetDatabaseName4HourlyReport) + '.sys.schemas where name = @Name';
        exec sp_executesql @tsql , N'@cnt INT OUTPUT, @Name VARCHAR(256)', @cnt = @TmpCnt OUTPUT, @Name = @TargetSchemaName4HourlyReport;
        if(@TmpCnt = 0)
        BEGIN
            RAISERROR('Schema %s does not exist in %s database',12,1,@TargetSchemaName4HourlyReport,@TargetDatabaseName4HourlyReport);
        END;

        -- Now we know that the database and schema exists.
        -- if the tablename is set ==> need to store data
        -- if not ... forgive it !

        if(@TargetTableName4HourlyReport is not null)
        BEGIN
            -- create table if not exist.
            SET @TmpCnt = null;

            SET @tsql = 'Use ' + QUOTENAME(@TargetDatabaseName4HourlyReport) + '; SELECT @ObjID = OBJECT_ID(''' + QUOTENAME(@TargetSchemaName4HourlyReport) + '.' + QUOTENAME(@TargetTableName4HourlyReport) + ''')';

            exec sp_executesql @tsql , N'@ObjID INT OUTPUT', @ObjID = @TmpCnt OUTPUT;

            if(@TmpCnt is null)
            BEGIN
                SET @tsql = 'CREATE TABLE ' + QUOTENAME(@TargetDatabaseName4HourlyReport) + '.' + QUOTENAME(@TargetSchemaName4HourlyReport) + '.' + QUOTENAME(@TargetTableName4HourlyReport) + '(' +@LineFeed +
                            '    LogDateWithHour  CHAR(10),' + @LineFeed +
                            '    DeadlockCount    INT' + @LineFeed +
                            ')';
                exec sp_executesql @tsql;
                if(@Debug = 1)
                BEGIN
                    SET @logMsg = 'Table ' + QUOTENAME(@TargetDatabaseName4HourlyReport) + '.' + QUOTENAME(@TargetSchemaName4HourlyReport) + '.' + QUOTENAME(@TargetTableName4HourlyReport) + ' created.';
                    RAISERROR(@logMsg,0,1);
                END;
            END;

            -- now the table is created.
            
            IF(@AppendMode4HourlyReport = 0)
            BEGIN
                SET @tsql = 'TRUNCATE TABLE ' + QUOTENAME(@TargetDatabaseName4HourlyReport) + '.' + QUOTENAME(@TargetSchemaName4HourlyReport) + '.' + QUOTENAME(@TargetTableName4HourlyReport);
                exec sp_executesql @tsql;
                
                if(@Debug = 1)
                BEGIN
                    SET @logMsg = 'Table ' + QUOTENAME(@TargetDatabaseName4HourlyReport) + '.' + QUOTENAME(@TargetSchemaName4HourlyReport) + '.' + QUOTENAME(@TargetTableName4HourlyReport) + ' truncated.';
                    RAISERROR(@logMsg,0,1);
                END;
            END;
            
            SET @tsql = 'INSERT INTO ' + QUOTENAME(@TargetDatabaseName4HourlyReport) + '.' + QUOTENAME(@TargetSchemaName4HourlyReport) + '.' + QUOTENAME(@TargetTableName4HourlyReport) + @LineFeed +
                        'SELECT LogDayShort + LogHour , count(*) FROM #ErrorLogsContent group by LogDayShort, LogHour';

            exec sp_executesql @tsql;
        END;
        -- Else, no storage

        /*
        -- TODO : check to see if possible to run it inside stored procedure (for code reuse)
        -- Storage - Target Table - Daily Report - PRE-Checks
        */
        
        -- ========== Daily Report

        if(@Debug = 1)
        BEGIN
            RAISERROR('Performing some checks before taking care of Daily report',0,1);
        END;
        
        if(@TargetDatabaseName4DailyReport is null)
        BEGIN
            SELECT @TargetDatabaseName4DailyReport = DB_NAME();
        END;

        if(DB_ID(@TargetDatabaseName4DailyReport) IS NULL) -- no QUOTENAME (it's quoted inside DB_ID)
        BEGIN
            RAISERROR('Database %s does not exist on server.',12,1,@TargetDatabaseName4DailyReport);
        END;

        if(@TargetSchemaName4DailyReport is null)
        BEGIN
            SELECT @TargetSchemaName4DailyReport =SCHEMA_NAME();
        END;

        -- check schema exists

        SET @tsql = 'SELECT @cnt = count(*) from ' + QUOTENAME(@TargetDatabaseName4DailyReport) + '.sys.schemas where name = @Name';
        exec sp_executesql @tsql , N'@cnt INT OUTPUT, @Name VARCHAR(256)', @cnt = @TmpCnt OUTPUT, @Name = @TargetSchemaName4DailyReport;
        if(@TmpCnt = 0)
        BEGIN
            RAISERROR('Schema %s does not exist in %s database',12,1,@TargetSchemaName4DailyReport,@TargetDatabaseName4DailyReport);
        END;

        -- Now we know that the database and schema exists.
        -- if the tablename is set ==> need to store data
        -- if not ... forgive it !

        if(@TargetTableName4DailyReport is not null)
        BEGIN
            -- create table if not exist.
            SET @TmpCnt = null;

            SET @tsql = 'Use ' + QUOTENAME(@TargetDatabaseName4DailyReport) + '; SELECT @ObjID = OBJECT_ID(''' + QUOTENAME(@TargetSchemaName4DailyReport) + '.' + QUOTENAME(@TargetTableName4DailyReport) + ''')';

            exec sp_executesql @tsql , N'@ObjID INT OUTPUT', @ObjID = @TmpCnt OUTPUT;

            if(@TmpCnt is null)
            BEGIN
                SET @tsql = 'CREATE TABLE ' + QUOTENAME(@TargetDatabaseName4DailyReport) + '.' + QUOTENAME(@TargetSchemaName4DailyReport) + '.' + QUOTENAME(@TargetTableName4DailyReport) + '(' +@LineFeed +
                            '    LogDateShort     CHAR(10),' + @LineFeed +
                            '    DeadlockCount    INT' + @LineFeed +
                            ')';
                exec sp_executesql @tsql;
                
                if(@Debug = 1)
                BEGIN
                    SET @logMsg = 'Table ' + QUOTENAME(@TargetDatabaseName4DailyReport) + '.' + QUOTENAME(@TargetSchemaName4DailyReport) + '.' + QUOTENAME(@TargetTableName4DailyReport) + ' created.';
                    RAISERROR(@logMsg,0,1);
                END;
            END;

            -- now the table is created.
            
            IF(@AppendMode4DailyReport = 0)
            BEGIN
                SET @tsql = 'TRUNCATE TABLE ' + QUOTENAME(@TargetDatabaseName4DailyReport) + '.' + QUOTENAME(@TargetSchemaName4DailyReport) + '.' + QUOTENAME(@TargetTableName4DailyReport);
                exec sp_executesql @tsql;
                
                if(@Debug = 1)
                BEGIN
                    SET @logMsg = 'Table ' + QUOTENAME(@TargetDatabaseName4DailyReport) + '.' + QUOTENAME(@TargetSchemaName4DailyReport) + '.' + QUOTENAME(@TargetTableName4DailyReport) + ' truncated.';
                    RAISERROR(@logMsg,0,1);
                END;
            END;
            
            SET @tsql = 'INSERT INTO ' + QUOTENAME(@TargetDatabaseName4DailyReport) + '.' + QUOTENAME(@TargetSchemaName4DailyReport) + '.' + QUOTENAME(@TargetTableName4DailyReport) + @LineFeed +
                        'SELECT LogDayShort , count(*) FROM #ErrorLogsContent group by LogDayShort';

            exec sp_executesql @tsql;
        END;
        -- Else, no storage
        
        -- if nothing have been given for storage, display it for Excel import
        if(@TargetTableName4HourlyReport is null AND @TargetTableName4DailyReport is null and @TargetTableName4OccurrencesReport is null)
        BEGIN 
            if(@Debug = 1)
            BEGIN
                RAISERROR('As no storage option have been provided, let''s display data for MS Excel (or equivalent) importation.',0,1);
            END;
    
            SELECT 
                LogDayShort as LogDayShort, '="' + CONVERT(VARCHAR(2),(LogHour)) + '"' as LogHourFormattedForExcel, count(*) as DeadlockCount
            from #ErrorLogsContent 
            group by LogDayShort, LogHour
            order by 1,2;
        END;
        
        if(@Debug = 1)
        BEGIN
            RAISERROR('Performing cleanups.',0,1);
        END;

        IF(OBJECT_ID('tempdb..#AllErrorLogs') IS NOT NULL)
        BEGIN
            exec sp_executesql N'DROP TABLE #AllErrorLogs';
        END;

        IF(OBJECT_ID('tempdb..#ErrorLogsContent') IS NOT NULL)
        BEGIN
            exec sp_executesql N'DROP TABLE #ErrorLogsContent';
        END;
    END TRY
    BEGIN CATCH
        -- Execute error retrieval routine.
        SELECT 
        ERROR_NUMBER() AS ErrorNumber,
        ERROR_SEVERITY() AS ErrorSeverity,
        ERROR_STATE() AS ErrorState,
        ERROR_PROCEDURE() AS ErrorProcedure,
        ERROR_LINE() AS ErrorLine,
        ERROR_MESSAGE() AS ErrorMessage;

        -- Test XACT_STATE:
            -- If 1, the transaction is committable.
            -- If -1, the transaction is uncommittable and should 
            --     be rolled back.
            -- XACT_STATE = 0 means that there is no transaction and
            --     a commit or rollback operation would generate an error.

        -- Test whether the transaction is uncommittable.
        IF (XACT_STATE()) = -1
        BEGIN
            PRINT
                N'The transaction is in an uncommittable state.' +
                'Rolling back transaction.'
            ROLLBACK TRANSACTION;
        END;

        -- Test whether the transaction is committable.
        IF (XACT_STATE()) = 1
        BEGIN
            PRINT
                N'The transaction is committable.' +
                'Committing transaction.'
            COMMIT TRANSACTION;   
        END;    
        END CATCH 
END;
GO

IF @@ERROR = 0
    PRINT '   PROCEDURE altered.';
ELSE
BEGIN
    PRINT '   Error while trying to alter procedure';
    RETURN;
END ;
GO 

PRINT '--------------------------------------------------------------------------------------------------------------';
PRINT '' ;
GO

/*
Query get logs of interest:

-- Get 
WITH LogDates(ArchiveID,FirstLogDate)
AS (
	select l2.ArchiveID , l1.LastLogDate
	From #AllErrorLogs l1 inner join #AllErrorLogs l2 on l1.ArchiveID = l2.ArchiveID + 1
) 
SELECT 
	el.ArchiveID, 
	ld.FirstLogDate, 
	el.LastLogDate
		
FROM #AllErrorLogs el
LEFT JOIN
	LogDates ld
ON el.ArchiveID = ld.ArchiveID
WHERE 
	1 = CASE
		WHEN (@StartDatePivot IS NULL AND @EndDatePivot IS NULL) 
			-- everything
		THEN 1
		WHEN (	@StartDatePivot IS NULL 
			AND @EndDatePivot IS NOT NULL 
		) 
			-- everything until
		THEN 
			CASE WHEN(
					@EndDatePivot BETWEEN ISNULL(ld.FirstLogDate,'1/1/1900 12:00:00') AND el.LastLogDate
				OR	@EndDatePivot >= el.LastLogDate
			)
				THEN 1
			ELSE 0
			END
		WHEN (	@StartDatePivot IS NOT NULL 
			AND @EndDatePivot IS NULL 
		)
			-- everything from
		THEN 
			CASE WHEN(
					@StartDatePivot BETWEEN ISNULL(ld.FirstLogDate,'1/1/1900 12:00:00') AND el.LastLogDate
				OR	@StartDatePivot <= ld.FirstLogDate
			)
				THEN 1
			ELSE 0
			END
		ELSE
			-- interval provided
			CASE WHEN(
				(
					@StartDatePivot BETWEEN ISNULL(ld.FirstLogDate,'1/1/1900 12:00:00') AND el.LastLogDate
				OR	@StartDatePivot <= ld.FirstLogDate
				)
				AND (
					@EndDatePivot BETWEEN ISNULL(ld.FirstLogDate,'1/1/1900 12:00:00') AND el.LastLogDate
				OR	@EndDatePivot >= el.LastLogDate
				)
			)
				THEN 1
			ELSE 0
			END
	END
;
*/