/*requires Schema.Monitoring.sql*/

DECLARE @ProcedureSchema NVARCHAR(256);
DECLARE @ProcedureName   NVARCHAR(256);

SET @ProcedureSchema = 'Monitoring' ;
SET @ProcedureName = 'CollectDeadlockInformation' ;

RAISERROR('-----------------------------------------------------------------------------------------------------------------',0,1);
RAISERROR('PROCEDURE [%s].[%s]',0,1,@ProcedureSchema,@ProcedureName);

IF  NOT EXISTS (SELECT 1 FROM sys.objects WHERE object_id = OBJECT_ID(N'[' + @ProcedureSchema + N'].[' + @ProcedureName +  N']') AND type in (N'P'))
BEGIN
    BEGIN TRY
        EXECUTE ('CREATE Procedure [' + @ProcedureSchema + '].[' + @ProcedureName +  '] ( ' +
                ' @ServerName    varchar(512), ' +
                ' @DbName    varchar(50) ' +
                ') ' +
                'AS ' +
                'BEGIN ' +
                '   SELECT ''Not implemented'' ' +
                'END')
    END TRY
    BEGIN CATCH
        PRINT '   Error while trying to create procedure'
        RETURN
    END CATCH

    PRINT '   PROCEDURE created.'
END
GO

ALTER PROCEDURE [Monitoring].[CollectDeadlockInformation] (
    @OutputType                     VARCHAR(16)     = 'NONE', -- Possible values: NONE, SCHEMA, TABLE
    @EventSessionName               VARCHAR(256)    = 'system_health',
    @EventSessionTargetType         VARCHAR(32)     = NULL,
    @UseApplicationParamsTable      BIT             = 1,
    @ApplicationName                VARCHAR(256)    = '$(SolutionName)',
    @LastCollectionParameterName    VARCHAR(256)    = 'DeadlockEvents:LastCollectionTime',
    @StartDatePivot		            DATETIME2       = NULL, 
    @EndDatePivot		            DATETIME2       = NULL,
    @OutputDatabaseName             SYSNAME         = NULL,
    @OutputSchemaName               SYSNAME         = NULL,
    @OutputTableName                SYSNAME         = NULL,
    @OutputAppendMode               BIT             = 1, -- set it to 0 to tell procedure to truncate table before inserting
    @Debug                          BIT             = 0,
    @_NoEventSessionNameCheck       BIT             = 0
)
AS
/*
  ===================================================================================
    DESCRIPTION:
        This procedure will collect information about deadlocks that occurred on a 
        given instance.
        It can use different data sources:
            - Default Extended Events Session "system_health"
            - Any other Extended Events Session
            
        We can ask for this procedure to limit its collection to a given time interval.
        
        Alternately, we can tell this procedure to use an ApplicationParams table in which
        it would find the value of a parameter that corresponds to its last successful execution time

    PARAMETERS:

    REQUIREMENTS:
        Utils.CheckTableExists stored procedure
        Common.ApplicationParams table
        Common.getApplicationParam stored procedure

	LIMITATION
		Only works on SQL Server 2008+

    EXAMPLE USAGE :
    
    
        Example 1: Collect and store information from "system_health" session using Application Params table
    
            EXEC [Monitoring].[CollectDeadlockInformation] 
                                @OutputType                 = 'NONE',
                                @UseApplicationParamsTable  = 1,
                                @ApplicationName            = 'Day2Day Database Management',
                                @OutputDatabaseName         = 'DBA',
                                @OutputSchemaName           = 'dbo',
                                @OutputTableName            = 'DeadlockHistory',
                                @Debug = 1
            ;
    
        Example 2: Collect and store information from "Collect-Deadlock" session using Application Params table
        
    
            EXEC [Monitoring].[CollectDeadlockInformation] 
                                @OutputType                 = 'NONE',
                                @EventSessionName           = 'Collect-Deadlock',
                                @UseApplicationParamsTable  = 1,
                                @ApplicationName            = 'Day2Day Database Management',
                                @OutputDatabaseName         = 'DBA',
                                @OutputSchemaName           = 'dbo',
                                @OutputTableName            = 'DeadlockHistory',
                                @Debug = 1
            ;

        Example 3: Get the list of deadlocks from system_health session 
        
            EXEC [Monitoring].[CollectDeadlockInformation] 
                    @OutputType                 = 'TABLE',
                    @EventSessionName           = 'system_health',
                    @UseApplicationParamsTable  = 0,
                    @Debug = 1
            ;
        
        Example 4: Collect and store deadlock information from system_health session but truncate table
            EXEC [Monitoring].[CollectDeadlockInformation] 
                                @OutputType                 = 'NONE',
                                @EventSessionName           = 'system_health',
                                @UseApplicationParamsTable  = 0,
                                @OutputDatabaseName         = 'DBA',
                                @OutputSchemaName           = 'dbo',
                                @OutputTableName            = 'DeadlockHistory',
                                @OutputAppendMode           = 0,
                                @Debug                      = 1
            ;
            
		SELECT * FROM DBA.dbo.DeadlockHistory order by DateStamp desc
		select * From Common.ApplicationParams where ParamName Like '%Deadlock%'
		SELECT SYSDATETIME(),DATEADD(MI,1,SYSDATETIME())
  ===================================================================================
*/

BEGIN
    SET NOCOUNT ON;
    DECLARE @tsql                   nvarchar(max);
    DECLARE @tsql_EventsData        nvarchar(max);
    DECLARE @StorageTsql            nvarchar(max);
    DECLARE @LineFeed               CHAR(2);
    DECLARE @ParamValidationFailed  BIT;
    DECLARE @MSSqlVersionMajor      INT;
    DECLARE @tmpStr                 VARCHAR(MAX);
    DECLARE @TmpCnt                 BIGINT;
    DECLARE @ExecRet                INT;
    DECLARE @LastExecTimeStr        VARCHAR(32);
    DECLARE @logMsg                 VARCHAR(4000);
    DECLARE @TruncateOutputTable    BIT;
    DECLARE @OutputSchemaValue      VARCHAR(MAX);
    DECLARE @EventFilePath          VARCHAR(MAX);
    DECLARE @EventMetaFilePath      VARCHAR(MAX);
    DECLARE @DeadLockGraphLookup    VARCHAR(4000);

    SELECT
        @tsql                   = '',
        @LineFeed               = CHAR(13) + CHAR(10),
        @ParamValidationFailed  = 0,
        @MSSqlVersionMajor      = (@@MICROSOFTVERSION / 0x1000000) & 0xff,
        @OutputDatabaseName     = ISNULL(@OutputDatabaseName,DB_NAME()),
        @OutputSchemaName       = ISNULL(@OutputSchemaName,SCHEMA_NAME()),
        @TruncateOutputTable    = CASE WHEN @OutputAppendMode = 1 THEN 0 ELSE 1 END,
        @OutputSchemaValue      =   '    DeadlockId         BIGINT IDENTITY(1,1) NOT NULL,' + @LineFeed +
                                    '    ServerName         VARCHAR(512),' + @LineFeed +
                                    '    AltServerId        BIGINT,' + @LineFeed +
                                    '    DateStamp          DATETIME2 NOT NULL,' + @LineFeed +
                                    '    DeadLockGraph      XML,' + @LineFeed +
                                    '    CollectionTime     DATETIME2 DEFAULT SYSDATETIME()' 
    ;

    if (@Debug = 1)
    BEGIN
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
        RAISERROR('-- Now running [Monitoring].[CollectDeadlockInformation] stored procedure.',0,1);
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
    END;

    if (@Debug = 1)
    BEGIN
        RAISERROR('-- Sanitizing input parameters',0,1); 
    END;
    
    IF(@EventSessionName = 'system_health')
    BEGIN   
        SET @EventSessionTargetType = 'ring_buffer';
    END;        
    
    IF(@EventSessionTargetType IS NULL AND @EventSessionName <> 'system_health')
    BEGIN
        IF(@Debug = 1)
        BEGIN
            RAISERROR('Defaulting empty @EventSessionTargetType parameter to "asynchronous_file_target"',0,1);
        END;
        SET @EventSessionTargetType = 'asynchronous_file_target';
    END;
    
    SELECT 
        @OutputType         = UPPER(@OutputType),
        @OutputTableName    = CASE WHEN LEN(LTRIM(RTRIM(@OutputTableName))) = 0 THEN NULL ELSE @OutputTableName END
    ;
    
    if (@Debug = 1)
    BEGIN
        RAISERROR('-- Performing parameter validation',0,1); 
    END;
    
    IF(@MSSqlVersionMajor < 10)
    BEGIN
        RAISERROR('SQL Server version cannot perform the actions of this procedure',12,1) WITH NOWAIT;
        RETURN;
    END;
    
    IF(@EventSessionTargetType = 'ring_buffer')
    BEGIN
        IF (@MSSqlVersionMajor = 10) 
        BEGIN
            SET @DeadLockGraphLookup = 'XEventData.XEvent.value(''(data/value)[1]'',''VARCHAR(MAX)'')';
        END;
        ELSE  -- only > 10
        BEGIN 
            SET @DeadLockGraphLookup = 'XEventData.XEvent.query(''(data/value/deadlock)[1]'')';
        END;
        
        SET @tsql_EventsData =  'WITH EventsData' + @LineFeed + 
                                'AS (' + @LineFeed +
                                    '    SELECT ' + @LineFeed +
                                    '	    CAST(target_data as xml) AS TargetData' + @LineFeed +
                                    '    FROM ' + @LineFeed +
                                    '	    sys.dm_xe_session_targets st' + @LineFeed +
                                    '    JOIN ' + @LineFeed +
                                    '	    sys.dm_xe_sessions s ' + @LineFeed +
                                    '    ON s.address = st.event_session_address' + @LineFeed +
                                    '    WHERE name   = ''' + @EventSessionName + '''' + @LineFeed +
                                    '    AND st.target_name = '''+ @EventSessionTargetType + '''' + @LineFeed +
                                ')' + @LineFeed +
                                'INSERT INTO #EventSessionItems' + @LineFeed +
                                'SELECT ' + @LineFeed +
                                --'    XEventData.XEvent.value(''@name'', ''varchar(100)'') as EventName,' + @LineFeed +
                                '    DATEADD(hh,DATEDIFF(hh,GETUTCDATE(),SYSDATETIME()),XEventData.XEvent.value(''@timestamp'', ''datetime2'')) as EventDate,' + @LineFeed +
                                '    CAST(' + @DeadLockGraphLookup + ' AS XML) AS DeadLockGraph ' + @LineFeed +
                                'FROM ' + @LineFeed +
                                '    EventsData' + @LineFeed +
                                'CROSS APPLY ' + @LineFeed +
                                '    TargetData.nodes(''//RingBufferTarget/event'') AS XEventData (XEvent)' + @LineFeed +
                                'WHERE ' + @LineFeed +
                                '    XEventData.XEvent.value(''@name'',''varchar(4000)'') = ''xml_deadlock_report''' + @LineFeed +
                                'AND DATEADD(hh,DATEDIFF(hh,GETUTCDATE(),SYSDATETIME()),XEventData.XEvent.value(''@timestamp'', ''datetime2'')) BETWEEN ISNULL(@StartDatePivot,''01/01/1900 12:00:00'') AND ISNULL(@EndDatePivot,''01/01/2100 12:00:00'')' + @LineFeed +
                                ';'
                                ;
    END;
    ELSE IF(@EventSessionTargetType = 'asynchronous_file_target')
    BEGIN
        -- Thanks to the author of https://www.sqlskills.com/blogs/jonathan/an-xevent-a-day-6-of-31-targets-week-asynchronous_file_target/ 
        -- it helped me a lot here :)
        
        -- Getting back event file path
        SELECT 
            @EventFilePath = LEFT(column_value, LEN(column_value) - CHARINDEX('.', REVERSE(column_value))) + '*' + RIGHT(column_value, CHARINDEX('.', REVERSE(column_value))-1)
        FROM 
            sys.dm_xe_sessions s
        JOIN 
            sys.dm_xe_session_object_columns soc
        ON s.address = soc.event_session_address
        WHERE 
            s.name          = @EventSessionName
        AND soc.object_name = @EventSessionTargetType
        AND soc.column_name = 'filename'
        ;
        
        -- getting back metadata file path
        SELECT  
            @EventMetaFilePath = LEFT(column_value, LEN(column_value)-CHARINDEX('.', REVERSE(column_value))) + '*' + RIGHT(column_value, CHARINDEX('.', REVERSE(column_value))-1)
        FROM 
            sys.dm_xe_sessions s
        JOIN 
            sys.dm_xe_session_object_columns soc
        ON s.address = soc.event_session_address
        WHERE 
            s.name          = @EventSessionName
        AND soc.object_name = @EventSessionTargetType
        AND soc.column_name = ' metadatafile'
        ;
        
        -- Set the metadata filename if it is NULL to the log file name with xem extension
        SET @EventMetaFilePath = ISNULL(@EventMetaFilePath, LEFT(@EventFilePath, LEN(@EventFilePath)-CHARINDEX('*', REVERSE(@EventFilePath))) + '*xem') ;
        
        SET @tsql_EventsData =  'WITH EventsData' + @LineFeed + 
                                'AS (' + @LineFeed +
                                '    SELECT' + @LineFeed +
                                '        CAST(event_data AS xml) as TargetData' + @LineFeed +
                                '    FROM sys.fn_xe_file_target_read_file(N''' + @EventFilePath + ''', N''' + @EventMetaFilePath + ''', null, null)' + @LineFeed + 
                                ')' + @LineFeed +
                                'INSERT INTO #EventSessionItems' + @LineFeed +
                                'SELECT' + @LineFeed +
                                '    --n.value(''(@name)[1]'', ''varchar(100)'') AS event_name,' + @LineFeed +
                                '    DATEADD(hh,DATEDIFF(hh,GETUTCDATE(),SYSDATETIME()),n.value(''@timestamp'', ''datetime2'')) as EventDate,' + @LineFeed +
                                '    CAST(n.value(''(data/value)[1]'',''VARCHAR(MAX)'') AS xml) as DeadLockGraph' + @LineFeed +
                                'FROM EventsData ' + @LineFeed +
                                'CROSS APPLY ' + @LineFeed +
                                '    TargetData.nodes(''event'') as q(n)' + @LineFeed +
                                'WHERE ' + @LineFeed +
                                '    n.value(''(@name)[1]'', ''varchar(100)'') = ''xml_deadlock_report''' + @LineFeed +
                                'AND DATEADD(hh,DATEDIFF(hh,GETUTCDATE(),SYSDATETIME()),n.value(''@timestamp'', ''datetime2'')) BETWEEN ISNULL(@StartDatePivot,''01/01/1900 12:00:00'') AND ISNULL(@EndDatePivot,''01/01/2100 12:00:00'')' + @LineFeed +
                                ';'
                                ;
    END;
    ELSE
    BEGIN
        RAISERROR('Event session target type not managed at the moment',12,1);
        RETURN;
    END;
    
    
    IF(@UseApplicationParamsTable = 1 AND (@ApplicationName IS NULL OR LEN(@ApplicationName) = 0))
    BEGIN
        RAISERROR('Parameter @ApplicationName must be provided to use ApplicationParams table',12,1) WITH NOWAIT;
        RETURN;
    END;
    
    -- Output Table Creation (when necessary)
    if(@OutputTableName is not null)
    BEGIN
        EXEC @ExecRet = [Utils].[CheckTableExists] 
                                    @DatabaseName       = @OutputDatabaseName,
                                    @SchemaName         = @OutputSchemaName,
                                    @TableName          = @OutputTableName ,
                                    @RunCreateTable     = 1,
                                    @TableColumnsDef    = @OutputSchemaValue,
                                    @RunTruncateTable   = @TruncateOutputTable,
									@Debug              = @Debug
        ;
        
        IF(@ExecRet <> 0)
        BEGIN
            RAISERROR('Unable to carry on as table existence check failed',12,1);
            RETURN;
        END;
    END;
    
    IF(@OutputType NOT IN ('NONE','SCHEMA','TABLE'))
    BEGIN
        RAISERROR('[%s] is not a recognized option for @OutputType parameter',12,1,@OutputType) WITH NOWAIT;
        SET @ParamValidationFailed = 1;
    END;
    
    IF(@OutputType = 'SCHEMA')
    BEGIN
        SELECT @OutputSchemaValue as TableDescription;
        RETURN;
    END;
    
    IF(@EventSessionName IS NULL OR LEN(@EventSessionName) = 0)
    BEGIN 
        RAISERROR('Mandatory parameter @EventSessionName not provided',12,1) WITH NOWAIT;
        SET @ParamValidationFailed = 1;
    END;
    ELSE
    BEGIN
        IF(@_NoEventSessionNameCheck = 0)
        BEGIN
            SET @tsql = 'SELECT @cnt = COUNT(*) FROM sys.dm_xe_sessions WHERE name = @EventSessionName';
            
            exec @ExecRet = sp_executesql  
                                @tsql,
                                N'@cnt INT OUTPUT, @EventSessionName VARCHAR(256)',
                                @cnt                = @TmpCnt OUTPUT, 
                                @EventSessionName   = @EventSessionName
            ;
            
            IF(@ExecRet <> 0)
            BEGIN
                RAISERROR('Unable to check existence of extended event session',12,1) WITH NOWAIT;
                RETURN;
            END;
            
            -- check event session actually exists
            IF(@TmpCnt <> 1)
            BEGIN
                RAISERROR('Extended Event Session with name [%s] not found.',12,1,@EventSessionName) WITH NOWAIT;
                SET @ParamValidationFailed = 1;
            END;
        END;
    END;
    
    IF(@UseApplicationParamsTable = 1)
    BEGIN
        -- Checking ApplicationParams table actually exists 
        IF(OBJECT_ID('Common.ApplicationParams') IS NULL)
        BEGIN
            RAISERROR('ApplicationParams table not found',12,1) WITH NOWAIT;
            RETURN;
        END;
        
        -- Getting back value from application params table 
        BEGIN TRY
            SET @tsql = 'EXEC Common.getApplicationParam @ApplicationName = @AppName, @ParamName = @PName, @ParamValue = @PValue OUTPUT';
            EXEC sp_executesql 
                            @tsql, 
                            N'@AppName VARCHAR(128),@PName VARCHAR(64),@PValue VARCHAR(MAX) OUTPUT',
                            @AppName = @ApplicationName,
                            @PName   = @LastCollectionParameterName,
                            @PValue  = @tmpStr OUTPUT
            ;
            
            IF(@tmpStr IS NULL OR LEN(@tmpStr) = 0)
            BEGIN 
                RAISERROR('Value for [%s] parameter not found in Common.ApplicationParams table',12,1,@LastCollectionParameterName) WITH NOWAIT;
            END;
            
            SET @LastExecTimeStr = @tmpStr;
            SET @tmpStr = '';
        END TRY
        BEGIN CATCH 
            IF(@Debug = 1)
            BEGIN
                RAISERROR('-- Cannot adjust @StartDatePivot using ApplicationParams table',0,1) WITH NOWAIT;
            END;
        END CATCH
        
        SET @LastExecTimeStr = SUBSTRING(@LastExecTimeStr,1,23);
        
        IF(@LastExecTimeStr IS NOT NULL AND LEN(@LastExecTimeStr) > 0) -- then adjust @StartDatePivot
        BEGIN
            IF(LEN(@LastExecTimeStr) <> 23) /* Format should be 121, which means YYYY-MM-DD hh24:mi:ss.mmm */
            BEGIN
                RAISERROR('Invalid format for [%s] parameter in Common.ApplicationParams table. Got value [%s]',12,1,@LastCollectionParameterName,@tmpStr) WITH NOWAIT;
                SET @ParamValidationFailed = 1;
            END;
            ELSE
            BEGIN
                SET @StartDatePivot = DATEADD(MILLISECOND,1,CONVERT(DATETIME2,@LastExecTimeStr,121));
            END;
        END;
        
    END;
    
    -- Check time interval
    IF(@StartDatePivot IS NOT NULL AND @EndDatePivot IS NOT NULL AND DATEDIFF(MICROSECOND,@StartDatePivot,@EndDatePivot) < 0)
    BEGIN
        RAISERROR('Provided date interval is not acceptable: @EndDatePivot < @StartDatePivot',12,1) WITH NOWAIT;
        SET @ParamValidationFailed = 1;
    END;
    
    IF(@ParamValidationFailed = 1)
    BEGIN
        RAISERROR('Parameter validation failed',12,1) WITH NOWAIT;
        RETURN;
    END;
    
    if(@Debug = 1)
    BEGIN
        RAISERROR('',0,1) WITH NOWAIT;
        RAISERROR('-- -------------- ( Debug Info: Params List ) ---------------------------',0,1) WITH NOWAIT;
        RAISERROR('-- Output Type                        : %s',0,1,@OutputType) WITH NOWAIT;
        RAISERROR('-- Event Session Name                 : %s',0,1,@EventSessionName) WITH NOWAIT;
        RAISERROR('-- Event Session Target Type          : %s',0,1,@EventSessionTargetType) WITH NOWAIT;
        SET @logMsg = '-- Use ApplicationParams table ?      : ' + CASE WHEN @UseApplicationParamsTable = 1 THEN 'Yes' ELSE 'No' END;
        RAISERROR(@logMsg,0,1) WITH NOWAIT;
        RAISERROR('-- Application Name                   : %s',0,1,@ApplicationName) WITH NOWAIT;
        RAISERROR('-- Last Collection Parameter Name     : %s',0,1,@LastCollectionParameterName) WITH NOWAIT;
        
        SET @logMsg = '-- Time interval left born (StartDate): ' + CASE WHEN @StartDatePivot IS NULL THEN 'N/A' ELSE CONVERT(VARCHAR(32),@StartDatePivot,121) END;
        RAISERROR(@logMsg,0,1) WITH NOWAIT;
        
        SET @logMsg = '-- Time interval right born (EndDate) : ' + CASE WHEN @EndDatePivot IS NULL THEN 'N/A' ELSE CONVERT(VARCHAR(32),@EndDatePivot,121) END;
        RAISERROR(@logMsg,0,1) WITH NOWAIT;
        
        SET @logMsg = '-- Full Table Name for Output Table       : ' + CASE 
                                                                            WHEN @OutputTableName IS NULL THEN 'N/A' 
                                                                            ELSE QUOTENAME(@OutputDatabaseName) + '.' + QUOTENAME(@OutputSchemaName) + '.' + QUOTENAME(@OutputTableName) 
                                                                        END
                                                                    ;
        RAISERROR(@logMsg,0,1) WITH NOWAIT;
        SET @logMsg = '-- Truncate output table ?            : ' + CASE WHEN @OutputAppendMode = 0 THEN 'Yes' ELSE 'No' END;
        
        RAISERROR('-- -------------- ( End of Debug Info ) ---------------------------',0,1) WITH NOWAIT;
        RAISERROR('',0,1) WITH NOWAIT;
        RAISERROR('-- Collecting deadlock informations.',0,1) WITH NOWAIT;
    END;

    /* ===========================================================================================================
     * Actual procedure action beginning
    */    
        
    IF(OBJECT_ID('tempdb..#EventSessionItems') IS NOT NULL)
    BEGIN
        exec sp_executesql N'DROP TABLE #EventSessionItems';
    END;

    if(@Debug = 1)
    BEGIN
        RAISERROR('-- Existing temporary tables used in this procedure may have been dropped.',0,1) WITH NOWAIT;
    END;
    
    BEGIN TRY
        
        CREATE TABLE #EventSessionItems (
            DateStamp           DATETIME2,
            DeadLockGraph       XML
        );
        
        SET @tsql = @tsql_EventsData;
        
        IF(@Debug = 1)
        BEGIN        
            RAISERROR('--------------------------Next Query-----------------------------------',0,1)
            RAISERROR(@tsql,0,1);
            RAISERROR('-----------------------------------------------------------------------',0,1)
        END;
        EXEC sp_executesql 
                    @tsql , 
                    N'@StartDatePivot DATETIME, @EndDatePivot DATETIME', 
                    @StartDatePivot = @StartDatePivot , 
                    @EndDatePivot = @EndDatePivot
        ;

        IF(@Debug = 1)
        BEGIN
            SET @tsql = 'SELECT @cnt = count_big(*) FROM #EventSessionItems';
            EXEC sp_executesql @tsql , N'@cnt BIGINT OUTPUT',@cnt = @TmpCnt OUTPUT;
            
            SET @logMsg = 'Total number of rows of interest: ' + CONVERT(VARCHAR(32),@TmpCnt);
            RAISERROR(@logMsg,0,1) WITH NOWAIT;
        END;
        
        SET @StorageTsql =  'INSERT INTO ' + QUOTENAME(@OutputDatabaseName) + '.' + QUOTENAME(@OutputSchemaName) + '.' + QUOTENAME(@OutputTableName) + '(' + @LineFeed +
                            '    ServerName,DateStamp,DeadLockGraph' + @LineFeed +
                            ')' + @LineFeed +
                            'SELECT @@SERVERNAME,DateStamp,DeadLockGraph' + @LineFeed +
                            'FROM #EventSessionItems' + @LineFeed + 
                            'ORDER BY DateStamp ASC' + @LineFeed
                            ;
        
        SET @tsql = 'SELECT @dt = MAX(DateStamp) FROM #EventSessionItems' ;
        EXEC sp_executesql @tsql, N'@dt DATETIME2 OUTPUT', @dt = @EndDatePivot OUTPUT;    
    
        if(@OutputTableName is not null)
        BEGIN
            IF(@Debug = 1)
            BEGIN
                RAISERROR('Now storing collected data to history table',0,1) WITH NOWAIT;
            END;
            
            exec @ExecRet = sp_executesql @StorageTsql;
            
            IF(@ExecRet <> 0)
            BEGIN
                RAISERROR('Unable to store collected data',12,1) WITH NOWAIT;
            END;
            
            IF(@UseApplicationParamsTable = 1)
            BEGIN
                SET @LastExecTimeStr = CONVERT(VARCHAR(32),@EndDatePivot,121);
                SET @tmpStr =  'Last execution time for ' + OBJECT_SCHEMA_NAME(@@PROCID) + '.' + OBJECT_NAME(@@PROCID) + '.' + @LineFeed +
                               'Note: this time is the internal value this procedure got from Extended Events (changed to local time)' + @LineFeed +
                               '      BUT this could be different from actual date and time for the occurrence.'
                               ;
                               
                SET @tsql = 'EXEC [Common].[ApplicationParams_Upsert]' + @LineFeed + 
                            '                   @ApplicationName    = @AppName,' + @LineFeed +
                            '                   @ParamName          = @PName,' + @LineFeed +
                            '                   @ParamValue         = @PValue,' + @LineFeed +
                            '                   @DefaultValue       = @PValue,' + @LineFeed +
                            '                   @ValueCanBeNULL     = 1,' + @LineFeed +
                            '                   @isDepreciated      = 0,' + @LineFeed +
                            '                   @ParamDescription   = @PDesc' + @LineFeed +
                            ';'
                            ;
                            
                exec sp_executesql 
                            @tsql, 
                            N'@AppName VARCHAR(128),@PName VARCHAR(64),@PValue VARCHAR(max),@PDesc VARCHAR(MAX)', 
                            @AppName    = @ApplicationName , 
                            @PName      = @LastCollectionParameterName, 
                            @PValue     = @LastExecTimeStr,
                            @PDesc      = @tmpStr
                ;
            END;
        END;
    
        IF(@OutputType = 'TABLE')
        BEGIN
            SELECT 
                NULL as DeadlockId, @@SERVERNAME as ServerName, DateStamp, DeadlockGraph, SYSDATETIME() as CollectionTime
            FROM #EventSessionItems
            ORDER BY DateStamp ASC;
        END;
    
    END TRY
    BEGIN CATCH
        -- Execute error retrieval routine.
        SELECT 
            ERROR_NUMBER() AS ErrorNumber,
            ERROR_SEVERITY() AS ErrorSeverity,
            ERROR_STATE() AS ErrorState,
            ERROR_PROCEDURE() AS ErrorProcedure,
            ERROR_LINE() AS ErrorLine,
            ERROR_MESSAGE() AS ErrorMessage
        ;

        -- Test XACT_STATE:
            -- If 1, the transaction is committable.
            -- If -1, the transaction is uncommittable and should 
            --     be rolled back.
            -- XACT_STATE = 0 means that there is no transaction and
            --     a commit or rollback operation would generate an error.

        -- Test whether the transaction is uncommittable.
        IF (XACT_STATE()) = -1
        BEGIN
            PRINT
                N'The transaction is in an uncommittable state.' +
                'Rolling back transaction.'
            ROLLBACK TRANSACTION;
        END;

        -- Test whether the transaction is committable.
        IF (XACT_STATE()) = 1
        BEGIN
            PRINT
                N'The transaction is committable.' +
                'Committing transaction.'
            COMMIT TRANSACTION;   
        END;    
    END CATCH 
    
    if(@Debug = 1)
    BEGIN
        RAISERROR('Performing cleanups.',0,1);
    END;

    IF(OBJECT_ID('tempdb..#EventSessionItems') IS NOT NULL)
    BEGIN
        exec sp_executesql N'DROP TABLE #EventSessionItems';
    END;    
    
    if (@Debug = 1)
    BEGIN
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
        RAISERROR('-- Execution of [Monitoring].[CollectDeadlockInformation] completed.',0,1);
        RAISERROR('-- -----------------------------------------------------------------------------------------------------------------',0,1);
    END;


END
GO


IF (@@ERROR = 0)
BEGIN
    PRINT '   PROCEDURE altered.';
END
ELSE
BEGIN
    PRINT '   Error while trying to alter procedure';
    RETURN
END;
GO

RAISERROR('-----------------------------------------------------------------------------------------------------------------',0,1);
RAISERROR('',0,1);
GO

