/*requires Schema.Reporting.sql*/
/*requires Table.Reporting.ExecutionLog.sql*/

DECLARE @ProcedureSchema NVARCHAR(256);
DECLARE @ProcedureName   NVARCHAR(256);

SET @ProcedureSchema = 'Reporting' ;
SET @ProcedureName = 'LogNewExecution' ;

PRINT '-----------------------------------------------------------------------------------------------------------------';
PRINT 'PROCEDURE [' + @ProcedureSchema + '].[' + @ProcedureName +  ']';

IF  NOT EXISTS (SELECT 1 FROM sys.objects WHERE object_id = OBJECT_ID(N'[' + @ProcedureSchema + N'].[' + @ProcedureName +  N']') AND type in (N'P'))
BEGIN
    BEGIN TRY
        EXECUTE ('CREATE Procedure [' + @ProcedureSchema + '].[' + @ProcedureName +  '] ( ' +
                ' @ServerName    varchar(512), ' +
                ' @DbName    varchar(50) ' +
                ') ' +
                'AS ' +
                'BEGIN ' +
                '   SELECT ''Not implemented'' ' +
                'END')
    END TRY
    BEGIN CATCH
        PRINT '   Error while trying to create procedure'
        RETURN
    END CATCH

    PRINT '   PROCEDURE created.'
END
GO

ALTER PROCEDURE [Reporting].[LogNewExecution] (
    @ReportClass                    VARCHAR(512) = NULL,
    @ReportName                     VARCHAR(512),
    @StartTime                      DATETIME     = NULL,
    @OutputType                     VARCHAR(32),
    @OutputPath                     VARCHAR(MAX) = NULL,
    @ReportId                       BIGINT OUTPUT,
    @Debug                          BIT = 0
)
AS
/*
  ===================================================================================
    DESCRIPTION:
        Create a new record in Reporting.ExecutionLog table and returns via its output
        parameter the value for ReportId of this record.
        
    PARAMETERS:

    REQUIREMENTS:

    EXAMPLE USAGE :
		DECLARE @ReportId BIGINT;

        EXEC [Reporting].[LogNewExecution] 
					@ReportName = 'TEST - Execution',
					@ReportClass = NULL,
					@ReportId	 = @ReportId OUTPUT,
                    @OutputType  = 'TABLE',
                    @OutputPath  = '[master].[dbo].[ThisIsATestTbl]',
					@Debug = 1
		;

		-- update details (Output type, report parameters)

		update Reporting.ExecutionLog
		set Outcome = 'SUCCESS'
		WHERE ReportId = @ReportId
		;

		SELECT * From Reporting.ExecutionLog 
		where ReportId = @ReportId
		;
  ===================================================================================
*/
BEGIN
    SET NOCOUNT ON;
    DECLARE @tsql               nvarchar(max);
    DECLARE @LineFeed           CHAR(2);

    SELECT
        @tsql               = '',
        @LineFeed           = CHAR(13) + CHAR(10),
        @StartTime          = ISNULL(@StartTime,GETDATE())
    ;

    if (@Debug = 1)
    BEGIN
        PRINT '-- -----------------------------------------------------------------------------------------------------------------';
        PRINT '-- Now running [Reporting].[LogNewExecution] stored procedure.';
        PRINT '-- -----------------------------------------------------------------------------------------------------------------';
    END;
    
    IF(@ReportName IS NULL OR LEN(@ReportName) < 1)
    BEGIN
        RAISERROR('Report has no name provided.',12,1) WITH NOWAIT;
        RETURN;
    END;
    
    IF(@OutputType IS NULL OR LEN(@OutputType) < 1)
    BEGIN
        RAISERROR('No output type provided.',12,1) WITH NOWAIT;
        RETURN;
    END;

    INSERT into [Reporting].[ExecutionLog] (
        ReportClass,ReportName,SQLLogin,ClientNetAddress,ClientHostName,ClientProgramName,StartTime, OutputType, OutputPath
    )
    SELECT
        @ReportClass,
        @ReportName,
        s.login_name,
        c.client_net_address,
        s.[host_name],
        s.program_name,
        @StartTime,
        @OutputType,
        @OutputPath
    from sys.dm_exec_requests r
    inner join sys.dm_exec_sessions s
    on r.session_id = s.session_id
    left join sys.dm_exec_connections c
    on r.session_id = c.session_id
    WHERE r.session_id = @@spid ;
    
    SET @ReportId = @@IDENTITY;

    if (@Debug = 1)
    BEGIN
        PRINT '-- -----------------------------------------------------------------------------------------------------------------';    
        PRINT '-- Execution of [Reporting].[LogNewExecution] completed.';
        PRINT '-- -----------------------------------------------------------------------------------------------------------------';
    END;
END
GO


IF (@@ERROR = 0)
BEGIN
    PRINT '   PROCEDURE altered.';
END
ELSE
BEGIN
    PRINT '   Error while trying to alter procedure';
    RETURN
END;
GO

PRINT '-----------------------------------------------------------------------------------------------------------------';
PRINT '';
GO

