USE [MSSQL_E11_TemporalTables]
GO
--\
---) For the demo, the virtualization layer is slightly different from a real life scenario.
---) Normally you would create separate databases for the Staging Area and the PSA, so you could do
---) the connection swap as explained here: 
---) http://www.hansmichiels.com/2017/02/18/using-a-persistent-staging-area-what-why-and-how/
---) Normally you would also either have a normal history table or a temporal one, but not both.
---) As I now have three objects that would like to have the virtual name [stg].[Customer], I use a 
---) suffix for the PSA versions, as this is workable for the demo.
---) So:  
---) [stg].[Customer]: view on the normal [stg_internal].[Customer] table.
---) [stg].[Customer_H]: view on the [psa].[Customer_History] table.
---) [stg].[Customer_TH]: view on the [psa].[Customer_Temporal] table.
--/

-------------- [Customer] --------------
IF OBJECT_ID('[stg].[Customer]', 'V') IS NOT NULL DROP VIEW [stg].[Customer];
SET ANSI_NULLS ON
SET QUOTED_IDENTIFIER ON
GO

CREATE VIEW [stg].[Customer]
AS
/*
==========================================================================================
Author:      Hans Michiels
Create date: 15-FEB-2017
Description: Virtualization view in order to be able to to full reloads from the PSA.
==========================================================================================
*/
SELECT        
    [CustomerID],
    [FirstName],
    [Initials],
    [MiddleName],
    [SurName],
    [DateOfBirth],
    [Gender],
    [SocialSecurityNumber],
    [Address],
    [PostalCode],
    [Residence],
    [StateOrProvince],
    [Country],
    [RowHash],
    [SessionStartDts]
FROM
    [stg_internal].[Customer]
GO


-------------- [Customer_H] --------------
IF OBJECT_ID('[stg].[Customer_H]', 'V') IS NOT NULL DROP VIEW [stg].[Customer_H];
SET ANSI_NULLS ON
SET QUOTED_IDENTIFIER ON
GO

CREATE VIEW [stg].[Customer_H]
AS
/*
==========================================================================================
Author:      Hans Michiels
Create date: 15-FEB-2017
Description: Virtualization view in order to be able to to full reloads from the PSA.
==========================================================================================
*/
SELECT        
    [CustomerID],
    [FirstName],
    [Initials],
    [MiddleName],
    [SurName],
    [DateOfBirth],
    [Gender],
    [SocialSecurityNumber],
    [Address],
    [PostalCode],
    [Residence],
    [StateOrProvince],
    [Country],
    [RowHash],
    [SessionStartDts]
FROM    
    [psa].[PointInTime] AS pit
JOIN
    [psa].[Customer_History] AS hist
    ON  hist.EffectiveStartDts <= pit.CurrentPointInTime
    AND hist.EffectiveEndDts > pit.CurrentPointInTime 
GO


-------------- [Customer_H] --------------
IF OBJECT_ID('[stg].[Customer_TH]', 'V') IS NOT NULL DROP VIEW [stg].[Customer_TH];
SET ANSI_NULLS ON
SET QUOTED_IDENTIFIER ON
GO

CREATE VIEW [stg].[Customer_TH]
AS
/*
==========================================================================================
Author:      Hans Michiels
Create date: 15-FEB-2017
Description: Virtualization view in order to be able to to full reloads from the PSA.
==========================================================================================
*/

SELECT        
    hist.[CustomerID],
    hist.[FirstName],
    hist.[Initials],
    hist.[MiddleName],
    hist.[SurName],
    hist.[DateOfBirth],
    hist.[Gender],
    hist.[SocialSecurityNumber],
    hist.[Address],
    hist.[PostalCode],
    hist.[Residence],
    hist.[StateOrProvince],
    hist.[Country],
    hist.[RowHash],
    hist.[SessionStartDts]
FROM    
    [psa].[PointInTime] AS pit
JOIN
    [psa].[Customer_Temporal] FOR SYSTEM_TIME ALL AS hist
    -- "FOR SYSTEM_TIME AS OF" does only work with a constant value or variable, 
    -- not by using a column from a joined table, e.g. pit.[CurrentPointInTime]
    -- So unfortunately we have to select all rows, and then do the date logic ourselves
    --
    -- Under the hood, a temporal table uses EXCLUSIVE enddating:
    -- the enddate of a row is equal to the startdate of the rows that replaces it.
    -- Therefore we can not use BETWEEN, as this includes the enddatetime.
    ON  hist.EffectiveStartDts <= pit.CurrentPointInTime
    AND hist.EffectiveEndDts > pit.CurrentPointInTime 
    -- By the way, there are more ways to do this, you could also use a CROSS JOIN and
    -- a WHERE clause here, instead doing the datetime filtering in the join.
GO



-------------- [Timeline_H] --------------
IF OBJECT_ID('[psa].[Timeline_H]', 'V') IS NOT NULL DROP VIEW [psa].[Timeline_H];
SET ANSI_NULLS ON
SET QUOTED_IDENTIFIER ON
GO

CREATE VIEW [psa].[Timeline_H]
AS
/*
==========================================================================================
Author:      Hans Michiels
Create date: 15-FEB-2017
Description: View used for time travelling.
==========================================================================================
*/
SELECT TOP 2147483647      
    CONVERT(VARCHAR(30), alltables_timeline.[SessionStartDts], 126) AS [SessionStartDtsString],
    -- If the point in time is the maxumum value of the [EffectiveStartDts] of the applicable [SessionStartDts]
    -- you will select all rows that were effective/valid after this session/load.
    CONVERT(VARCHAR(30), MAX(alltables_timeline.[EffectiveStartDts]), 126) AS [PointInTimeDtsString]
FROM
    (
    SELECT DISTINCT 
        subh.[SessionStartDts], 
        subh.[EffectiveStartDts]
    FROM
        [psa].[Customer_History] subh WITH (READPAST)
    -- UNION MORE STAGING TABLE HERE WHEN APPLICABLE
    ) alltables_timeline
GROUP BY
    CONVERT(VARCHAR(30), alltables_timeline.[SessionStartDts], 126)
ORDER BY 
    CONVERT(VARCHAR(30), alltables_timeline.[SessionStartDts], 126)
GO

-------------- [Timeline_TH] --------------
IF OBJECT_ID('[psa].[Timeline_TH]', 'V') IS NOT NULL DROP VIEW [psa].[Timeline_TH];
SET ANSI_NULLS ON
SET QUOTED_IDENTIFIER ON
GO

CREATE VIEW [psa].[Timeline_TH]
AS
/*
==========================================================================================
Author:      Hans Michiels
Create date: 15-FEB-2017
Description: View used for time travelling.
==========================================================================================
*/
SELECT TOP 2147483647    
    CONVERT(VARCHAR(30), alltables_timeline.[SessionStartDts], 126) AS [SessionStartDtsString],
    -- If the point in time is the maxumum value of the [EffectiveStartDts] of the applicable [SessionStartDts]
    -- you will select all rows that were effective/valid after this session/load.
    CONVERT(VARCHAR(30), MAX(alltables_timeline.[EffectiveStartDts]), 126) AS [PointInTimeDtsString]
FROM
    (
    SELECT DISTINCT
        subh.[SessionStartDts], 
        subh.[EffectiveStartDts]
    FROM
        [psa].[Customer_Temporal] FOR SYSTEM_TIME ALL AS subh WITH (READPAST)
    -- UNION MORE STAGING TABLE HERE WHEN APPLICABLE
    ) alltables_timeline
GROUP BY
    CONVERT(VARCHAR(30), alltables_timeline.[SessionStartDts], 126)
ORDER BY 
    CONVERT(VARCHAR(30), alltables_timeline.[SessionStartDts], 126)
GO
