-- Create a database named [MSSQL_E11_TemporalTables] first.

USE [MSSQL_E11_TemporalTables]
GO
--\
---) For the sake of simplicity I create all objects within one database.
---) Still the concept of putting each layer in it's own database schema is followed.
---) For objects only used inside a layer I use schemanames suffixed with _internal.
--/

--\
---) Create a schema 'stg' for the Staging Area, 
---) if this schema does not exist yet.
--/
IF NOT EXISTS(SELECT 1 FROM sys.schemas where name = N'stg')
BEGIN
    -- Must use dynamic SQL otherwise syntax error occurs.
    EXEC('CREATE SCHEMA [stg] authorization [dbo]')
END
GO

--\
---) Create a schema 'stg' for the internally used objects in the Staging Area, 
---) if this schema does not exist yet.
--/
IF NOT EXISTS(SELECT 1 FROM sys.schemas where name = N'stg_internal')
BEGIN
    -- Must use dynamic SQL otherwise syntax error occurs.
    EXEC('CREATE SCHEMA [stg_internal] authorization [dbo]')
END
GO

--\
---) Create a schema 'psa' for the Persistent Staging Area, 
---) if this schema does not exist yet.
--/
IF NOT EXISTS(SELECT 1 FROM sys.schemas where name = N'psa')
BEGIN
    -- Must use dynamic SQL otherwise syntax error occurs.
    EXEC('CREATE SCHEMA [psa] authorization [dbo]')
END
GO

--\
---) Create a schema 'psa' for the internally used tables, 
---) if this schema does not exist yet.
--/
IF NOT EXISTS(SELECT 1 FROM sys.schemas where name = N'psa_internal')
BEGIN
    -- Must use dynamic SQL otherwise syntax error occurs.
    EXEC('CREATE SCHEMA [psa_internal] authorization [dbo]')
END
GO

--\
---) Create a schema 'test' for test objects, 
---) if this schema does not exist yet.
--/
IF NOT EXISTS(SELECT 1 FROM sys.schemas where name = N'test')
BEGIN
    -- Must use dynamic SQL otherwise syntax error occurs.
    EXEC('CREATE SCHEMA [test] authorization [dbo]')
END
GO



--\
---) Drop the previous version of the table.
--/
IF OBJECT_ID('[stg_internal].[Customer]', 'U') IS NOT NULL
BEGIN
    DROP TABLE [stg_internal].[Customer];
END

IF OBJECT_ID('[psa].[Customer_Temporal]', 'U') IS NOT NULL
BEGIN
    ALTER TABLE [psa].[Customer_Temporal] SET (SYSTEM_VERSIONING = OFF);
    DROP TABLE [psa].[Customer_Temporal];
    DROP TABLE [psa].[Customer_TemporalHistory];
END
GO

IF OBJECT_ID('[psa].[Customer_History]', 'U') IS NOT NULL
BEGIN
    DROP TABLE [psa].[Customer_History];
END

--\
---) Now create the staging table
---) The schema [stg_internal] is used for this table, 
---) so I can put a view with the same name on top of it in the [stg] schema.
--/
CREATE TABLE [stg_internal].[Customer]
(    
  [CustomerID] INT NOT NULL,
  [FirstName] NVARCHAR(20) NULL,
  [Initials] NVARCHAR(20) NULL,
  [MiddleName] NVARCHAR(20) NULL,
  [SurName] NVARCHAR(50) NOT NULL,
  [DateOfBirth] DATE NOT NULL,
  [Gender] CHAR(1) NOT NULL,
  [SocialSecurityNumber] CHAR(12) NOT NULL,
  [Address] NVARCHAR(60) NOT NULL,
  [PostalCode] CHAR(10) NULL,
  [Residence] NVARCHAR(60) NULL,
  [StateOrProvince] NVARCHAR(20) NULL,
  [Country] NVARCHAR(60) NULL,
  [RowHash] BINARY(16), 
  [SessionStartDts] DATETIME2(2) NOT NULL, 
  CONSTRAINT [PK_Customer]
    PRIMARY KEY CLUSTERED  ([CustomerID] ASC),
)
GO

--\
---) Now create the history table to be used as backing table for the temporal table, 
---) so we can tweak it for optimal performance.
---) Please note that I use the datatype DATETIME2(2) because it
---) uses 6 bytes storage, whereas DATETIME2(7) uses 8 bytes.
---) If the centiseconds precision of DATETIME2(2) is not enough
---) in your data warehouse, you can change it to DATETIME2(7).
--/
CREATE TABLE [psa].[Customer_TemporalHistory]
(    
  [CustomerID] INT NOT NULL,
  [FirstName] NVARCHAR(20) NULL,
  [Initials] NVARCHAR(20) NULL,
  [MiddleName] NVARCHAR(20) NULL,
  [SurName] NVARCHAR(50) NOT NULL,
  [DateOfBirth] DATE NOT NULL,
  [Gender] CHAR(1) NOT NULL,
  [SocialSecurityNumber] CHAR(12) NOT NULL,
  [Address] NVARCHAR(60) NOT NULL,
  [PostalCode] CHAR(10) NULL,
  [Residence] NVARCHAR(60) NULL,
  [StateOrProvince] NVARCHAR(20) NULL,
  [Country] NVARCHAR(60) NULL,
  [RowHash] BINARY(16), 
  [SessionStartDts] DATETIME2(2) NOT NULL, 
  [EffectiveStartDts] DATETIME2(2) NOT NULL, 
  [EffectiveEndDts] DATETIME2(2) NOT NULL
);
GO

--\
---) Add indexes to history table
--/
CREATE CLUSTERED COLUMNSTORE INDEX [IXCS_Customer_TemporalHistory]
  ON [psa].[Customer_TemporalHistory];

CREATE NONCLUSTERED INDEX [IXNC_Customer_TemporalHistory__EffectiveEndDts_EffectiveStartDts_CustomerID]
  ON [psa].[Customer_TemporalHistory] 
    ([EffectiveEndDts], [EffectiveStartDts], [CustomerID]);

GO
                   
--\
---) Now create the temporal table
--/
CREATE TABLE [psa].[Customer_Temporal]
(    
  [CustomerID] INT NOT NULL,
  [FirstName] NVARCHAR(20) NULL,
  [Initials] NVARCHAR(20) NULL,
  [MiddleName] NVARCHAR(20) NULL,
  [SurName] NVARCHAR(50) NOT NULL,
  [DateOfBirth] DATE NOT NULL,
  [Gender] CHAR(1) NOT NULL,
  [SocialSecurityNumber] CHAR(12) NOT NULL,
  [Address] NVARCHAR(60) NOT NULL,
  [PostalCode] CHAR(10) NULL,
  [Residence] NVARCHAR(60) NULL,
  [StateOrProvince] NVARCHAR(20) NULL,
  [Country] NVARCHAR(60) NULL,
  [RowHash] BINARY(16), 
  [SessionStartDts] DATETIME2(2) NOT NULL, 
  CONSTRAINT [PK_Customer_Temporal]
    PRIMARY KEY NONCLUSTERED  ([CustomerID] ASC), 
  [EffectiveStartDts] DATETIME2(2) GENERATED ALWAYS AS ROW START NOT NULL,
  [EffectiveEndDts] DATETIME2(2) GENERATED ALWAYS AS ROW END NOT NULL,
  PERIOD FOR SYSTEM_TIME ([EffectiveStartDts], [EffectiveEndDts]) 
)
WITH (SYSTEM_VERSIONING = ON (HISTORY_TABLE = [psa].[Customer_TemporalHistory])); 
GO

-- Add a few indexes.
CREATE CLUSTERED COLUMNSTORE INDEX [IXCS_Customer_Temporal]
  ON [psa].[Customer_Temporal];

CREATE NONCLUSTERED INDEX [IXNC_Customer_Temporal__EffectiveEndDts_EffectiveStartDts_CustomerID]
  ON [psa].[Customer_Temporal] 
    ([EffectiveEndDts], [EffectiveStartDts], [CustomerID]);

CREATE NONCLUSTERED INDEX [IXNC_Customer_Temporal__CustomerID_RowHash]
  ON [psa].[Customer_Temporal] 
    ([CustomerID], [RowHash]);
GO

--\
---) Create the 'PSA' staging table without using a temporal table
--/
CREATE TABLE [psa].[Customer_History]
(    
  [CustomerID] INT NOT NULL,
  [FirstName] NVARCHAR(20) NULL,
  [Initials] NVARCHAR(20) NULL,
  [MiddleName] NVARCHAR(20) NULL,
  [SurName] NVARCHAR(50) NOT NULL,
  [DateOfBirth] DATE NOT NULL,
  [Gender] CHAR(1) NOT NULL,
  [SocialSecurityNumber] CHAR(12) NOT NULL,
  [Address] NVARCHAR(60) NOT NULL,
  [PostalCode] CHAR(10) NULL,
  [Residence] NVARCHAR(60) NULL,
  [StateOrProvince] NVARCHAR(20) NULL,
  [Country] NVARCHAR(60) NULL,
  [RowHash] BINARY(16), 
  [SessionStartDts] DATETIME2(2) NOT NULL, 
  [EffectiveStartDts] DATETIME2(2) NOT NULL,
  [EffectiveEndDts] DATETIME2(2) NOT NULL,
  CONSTRAINT [PK_Customer_History]
    PRIMARY KEY NONCLUSTERED ([CustomerID] ASC, [EffectiveStartDts] ASC),
)
GO

--\
---) Add indexes to history table
--/
CREATE CLUSTERED COLUMNSTORE INDEX [IXCS_Customer_History]
  ON [psa].[Customer_History];

CREATE NONCLUSTERED INDEX [IXNC_Customer_History__EffectiveEndDts_EffectiveStartDts_CustomerID]
  ON [psa].[Customer_History] 
    ([EffectiveEndDts], [EffectiveStartDts], [CustomerID]);

CREATE NONCLUSTERED INDEX [IXNC_Customer_History__CustomerID_RowHash]
  ON [psa].[Customer_History] 
    ([CustomerID], [RowHash]);
GO

--\
---) Create table needed for time travelling.
--/
IF OBJECT_ID(N'[psa].[PointInTime]', 'U') IS NOT NULL
  DROP TABLE [psa].[PointInTime];

CREATE TABLE [psa].[PointInTime]
(
  [PointInTimeID] INT NOT NULL,
  [CurrentPointInTime] DATETIME2(2) NOT NULL,
  CONSTRAINT [CHK_psa_PointInTime_Ensure_Single_Row] CHECK ([PointInTimeID] = 1),
  CONSTRAINT [PK_psa_PointInTime]
    PRIMARY KEY CLUSTERED ([PointInTimeID] ASC),
)
GO
INSERT INTO [psa].[PointInTime]([PointInTimeID], [CurrentPointInTime])
VALUES (1, GETUTCDATE());